<?php
/**
 * PDF Report Generator
 *
 * used by the SAX parser to generate PDF reports from the XML report file.
 *
 * phpGedView: Genealogy Viewer
 * Copyright (C) 2002 to 2021  PGV Development Team.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @package PhpGedView
 * @subpackage Reports
 * @version $Id: class_reportpdf.php 7295 2021-04-03 10:43:57Z canajun2eh $
 * @link http://www.adobe.com/devnet/pdf/
 * @link http://www.tcpdf.org
 */

if (!defined('PGV_PHPGEDVIEW')) {
	header('HTTP/1.0 403 Forbidden');
	exit;
}

define('PGV_CLASS_REPORTPDF_PHP', '');

require_once PGV_ROOT."includes/classes/class_reportbase.php";
require_once PGV_ROOT."includes/tcpdf/tcpdf.php";

/**
* Main PGV Report Class for PDF
*
* @see PGVReportBase
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVReportBasePDF extends PGVReportBase {
	/**
	* PDF compression - Zlib extension is required
	* @var boolean const compression
	*/
	const compression = true;
	/**
	* If TRUE reduce the RAM memory usage by caching temporary data on filesystem (slower).
	* @var boolean const diskcache
	*/
	const diskcache = false;
	/**
	* TRUE means that the input text is unicode (PDF)
	* @var boolean const unicode
	*/
	const unicode = true;
	/**
	* A new object of the PGVRPDF class
	* @var PGVRPDF
	*/
	public $pdf;
	
	/**
	* PDF Setup - PGVReportBasePDF
	*
	* @package PhpGedView
	* @subpackage Reports
	*/
	function setup() {
		parent::setup();

		// Setup the PDF class with custom size pages because PGV supports more page sizes. If PGV sends an unknown size name then the default would be A4
		$this->pdf = new PGVRPDF($this->orientation, parent::unit, array($this->pagew, $this->pageh), self::unicode, $this->charset, self::diskcache);

		// Setup the PDF margins
		$this->pdf->setMargins($this->leftmargin, $this->topmargin, $this->rightmargin);
		$this->pdf->SetHeaderMargin($this->headermargin);
		$this->pdf->SetFooterMargin($this->footermargin);
		//Set auto page breaks
		$this->pdf->SetAutoPageBreak(true, $this->bottommargin);
		// Setup PDF compression
		$this->pdf->SetCompression(self::compression);
		// Setup RTL support
		$this->pdf->setRTL($this->rtl);
		// Set the document information
		// Only admin should see the version number
		$appversion = PGV_PHPGEDVIEW;
		if (PGV_USER_IS_ADMIN) {
			$appversion .= " ".PGV_VERSION_TEXT;
		}
		$this->pdf->SetCreator($appversion." (".parent::pgv_url.")");
		// Not implemented yet - PGVReportBase::setup()
//		$this->pdf->SetAuthor($this->rauthor);
		$this->pdf->SetTitle($this->title);
		$this->pdf->SetSubject($this->rsubject);
		$this->pdf->SetKeywords($this->rkeywords);

		$this->pdf->setReport($this);

		if ($this->showGenText) {
			// The default style name for Generated by.... is 'genby'
			$element = new PGVRCellPDF(0, 10, 0, "C", "", "genby", 1, ".", ".", 0, 0, "", "", true, true);
			$element->addText($this->generatedby);
			$element->setUrl(parent::pgv_url);
			$this->pdf->addFooter($element);
		}
	}

	/**
	* Add an element - PGVReportBasePDF
	* @param object|string &$element Object or string
	*
	* @package PhpGedView
	* @subpackage Reports
	*/
	function addElement($element) {
		if ($this->processing == "B") {
			return $this->pdf->addBody($element);
		} elseif ($this->processing == "H") {
			return $this->pdf->addHeader($element);
		} elseif ($this->processing == "F") {
			return $this->pdf->addFooter($element);
		}
		return 0;
	}

	function run() {
		global $download;

		$this->pdf->Body();
		header("Expires:");
		header("Pragma:");
		header("Cache-control:");
		if ($download == "") {
			$this->pdf->Output();
		} else {
			$this->pdf->Output("pgv_report_".basename($_REQUEST["report"], ".xml").".pdf", "D");
		}
		return;
	}

	/**
	* Clear the Header - PGVReportBasePDF
	* @package PhpGedView
	* @subpackage Reports
	*/
	function clearHeader() {
		$this->pdf->clearHeader();
	}

	/**
	* Clear the Page Header - PGVReportBasePDF
	* @package PhpGedView
	* @subpackage Reports
	*/
	function clearPageHeader() {
		$this->pdf->clearPageHeader();
	}

	/**
	* Create a new Cell object - PGVReportBasePDF
	*
	* @param int $width cell width (expressed in points)
	* @param int $height cell height (expressed in points)
	* @param mixed $border Border style
	* @param string $align Text alignement
	* @param string $bgcolor Background color code
	* @param string $style The name of the text style
	* @param int $ln Indicates where the current position should go after the call
	* @param mixed $top Y-position
	* @param mixed $left X-position
	* @param int $fill Indicates if the cell background must be painted (1) or transparent (0). Default value: 1
	* @param int $stretch Stretch carachter mode
	* @param string $bocolor Border color
	* @param string $tcolor Text color
	* @param boolean $reseth
	* @param boolean $padding=true
	* @return PGVRCellPDF
	* 
	* For Defaults and info
	* @see PGVRCell
	* @see PGVRCellSHandler()
	*/
	function createCell($width, $height, $border, $align, $bgcolor, $style, $ln, $top, $left, $fill, $stretch, $bocolor, $tcolor, $reseth, $padding) {
		return new PGVRCellPDF($width, $height, $border, $align, $bgcolor, $style, $ln, $top, $left, $fill, $stretch, $bocolor, $tcolor, $reseth, $padding);
	}

	/**
	* Create a new TextBox object - PGVReportBasePDF
	*
	* @param float $width Text box width
	* @param float $height Text box height
	* @param boolean $border
	* @param string $bgcolor Background color code in HTML
	* @param boolean $newline
	* @param mixed $left
	* @param mixed $top
	* @param boolean $pagecheck
	* @param string $style
	* @param boolean $fill
	* @param boolean $padding
	* @param boolean $reseth
	* @return PGVRTextBoxPDF
	*/
	function createTextBox($width, $height, $border, $bgcolor, $newline, $left, $top, $pagecheck, $style, $fill, $padding, $reseth) {
		return new PGVRTextBoxPDF($width, $height, $border, $bgcolor, $newline, $left, $top, $pagecheck, $style, $fill, $padding, $reseth);
	}

	/**
	* Create a new Text object- PGVReportBasePDF
	*
	* @param string $style The name of the text style
	* @param string $color HTML color code
	* @return PGVRTextPDF
	*/
	function createText($style, $color) {
		return new PGVRTextPDF($style, $color);
	}

	/**
	* Create a new Footnote object - PGVReportBasePDF
	* @param string $style Style name
	* @return PGVRFootnotePDF
	*/
	function createFootnote($style) {
		return new PGVRFootnotePDF($style);
	}

	/**
	* Create a new Page Header object - PGVReportBasePDF
	* @return PGVRPageHeaderPDF
	*/
	function createPageHeader() {
		return new PGVRPageHeaderPDF();
	}

	/**
	* Create a new image object - PGVReportBasePDF
	* @param string $file File name
	* @param mixed $x
	* @param mixed $y
	* @param int $w Image width
	* @param int $h Image height
	* @param string $align L:left, C:center, R:right or empty to use x/y
	* @param string $ln T:same line, N:next line
	* @return PGVRImagePDF
	*/
	function createImage($file, $x, $y, $w, $h, $align, $ln) {
		return new PGVRImagePDF($file, $x, $y, $w, $h, $align, $ln);
	}

	/**
	* Create a new line object - PGVReportBasePDF
	* @param mixed $x1
	* @param mixed $y1
	* @param mixed $x2
	* @param mixed $y2
	* @return PGVRLinePDF
	*/
	function createLine($x1, $y1, $x2, $y2) {
		return new PGVRLinePDF($x1, $y1, $x2, $y2);
	}

	/**
	* @return PGVRHtmlPDF
	*/
	function createHTML($tag, $attrs) {
		return new PGVRHtmlPDF($tag, $attrs);
	}
} //-- end PGVReport

/**
* PGV Report PDF Class
*
* This class inherits from the TCPDF class and is used to generate the PDF document
* @package PhpGedView
* @subpackage Reports
*/
class PGVRPDF extends TCPDF {
	/**
	* Array of elements in the header
	* @var array $headerElements
	*/
	public $headerElements = array();
	/**
	* Array of elements in the page header
	* @var array $pageHeaderElements
	*/
	public $pageHeaderElements = array();
	/**
	* Array of elements in the footer
	* @var array $footerElements
	*/
	public $footerElements = array();
	/**
	* Array of elements in the body
	* @var array $bodyElements
	*/
	public $bodyElements = array();
	/**
	* Array of elements in the footer notes
	* @var array $printedfootnotes
	*/
	public $printedfootnotes = array();
	/**
	* Currently used style name
	* @var string $currentStyle
	*/
	public $currentStyle = '';
	/**
	* The last cell height
	* @var int $lastCellHeight
	*/
	public $lastCellHeight = 0;
	/**
	* The largest font size within a PGVRTextBox
	* to calculate the height
	* @var int $largestFontHeight
	*/
	public $largestFontHeight = 0;
	/**
	* The last pictures page number
	* @var int $lastpicpage
	*/
	public $lastpicpage = 0;

	public $pgvreport;

	/**
	* PDF Header -PGVRPDF
	*/
	function Header() {
		foreach($this->headerElements as $element) {
			if (is_object($element)) {
				$element->render($this);
			} elseif (is_string($element) && $element=="footnotetexts") {
				$this->Footnotes();
			} elseif (is_string($element) && $element=="addpage") {
				$this->newPage();
			}
		}
		foreach($this->pageHeaderElements as $element) {
			if (is_object($element)) {
				$element->render($this);
			} elseif (is_string($element) && $element=="footnotetexts") {
				$this->Footnotes();
			} elseif (is_string($element) && $element=="addpage") {
				$this->newPage();
			}
		}
	}

	/**
	* PDF Body -PGVRPDF
	*/
	function Body() {
		$this->AddPage();
		foreach($this->bodyElements as $key => $element) {
			if (is_object($element)) {
				$element->render($this);
			} elseif (is_string($element) && $element=="footnotetexts") {
				$this->Footnotes();
			} elseif (is_string($element) && $element=="addpage") {
				$this->newPage();
			}
			// Delete used elements in hope to reduce 'some' memory usage
			unset($this->bodyElements[$key]);
		}
	}

	/**
	* PDF Footnotes -PGVRPDF
	*/
	function Footnotes() {
		foreach($this->printedfootnotes as $element) {
			if (($this->GetY() + $element->getFootnoteHeight($this)) > $this->getPageHeight()) {
				$this->AddPage();
			}
			$element->renderFootnote($this);
			if ($this->GetY() > $this->getPageHeight()) {
				$this->AddPage();
			}
		}
	}

	/**
	* PDF Footer -PGVRPDF
	*/
	function Footer() {
		foreach($this->footerElements as $element) {
			if (is_object($element)) {
				$element->render($this);
			} elseif (is_string($element) && $element=="footnotetexts") {
				$this->Footnotes();
			} elseif (is_string($element) && $element=="addpage") {
				$this->newPage();
			}
		}
	}

	/**
	* Add an element to the Header -PGVRPDF
	* @param object|string &$element
	* @return int The number of the Header elements
	*/
	function addHeader($element) {
		$this->headerElements[] = $element;
		return count($this->headerElements)-1;
	}

	/**
	* Add an element to the Page Header -PGVRPDF
	* @param object|string &$element
	* @return int The number of the Page Header elements
	*/
	function addPageHeader($element) {
		$this->pageHeaderElements[] = $element;
		return count($this->pageHeaderElements)-1;
	}

	/**
	* Add an element to the Body -PGVRPDF
	* @param object|string &$element
	* @return int The number of the Body elements
	*/
	function addBody($element) {
		$this->bodyElements[] = $element;
		return count($this->bodyElements)-1;
	}

	/**
	* Add an element to the Footer -PGVRPDF
	* @param object|string &$element
	* @return int The number of the Footer elements
	*/
	function addFooter($element) {
		$this->footerElements[] = $element;
		return count($this->footerElements)-1;
	}

	function removeHeader($index) {
		unset($this->headerElements[$index]);
	}

	function removePageHeader($index) {
		unset($this->pageHeaderElements[$index]);
	}

	function removeBody($index) {
		unset($this->bodyElements[$index]);
	}

	function removeFooter($index) {
		unset($this->footerElements[$index]);
	}

	/**
	* Clear the Header -PGVRPDF
	*/
	function clearHeader() {
		unset($this->headerElements);
		$this->headerElements = array();
	}

	/**
	* Clear the Page Header -PGVRPDF
	*/
	function clearPageHeader() {
		unset($this->pageHeaderElements);
		$this->pageHeaderElements = array();
	}

	function setReport($r) {
		$this->pgvreport = $r;
	}

	/**
	* Get the currently used style name -PGVRPDF
	* @return string
	*/
	function getCurrentStyle() {
		return $this->currentStyle;
	}

	/**
	* Setup a style for usage -PGVRPDF
	* @param string $s Style name
	*/
	function setCurrentStyle($s) {
		$this->currentStyle = $s;
		$style = $this->pgvreport->getStyle($s);
		$this->SetFont($style["font"], $style["style"], $style["size"]);
	}

	/**
	* Get the style -PGVRPDF
	* @param string $s Style name
	* @return array
	*/
	function getStyle($s) {
		if (!isset($this->pgvreport->PGVRStyles[$s])) {
			$s = $this->getCurrentStyle();
			$this->pgvreport->PGVRStyles[$s] = $s;
		}
		return $this->pgvreport->PGVRStyles[$s];
	}

	/**
	* Add margin when static horizontal position is used -PGVRPDF
	* RTL supported
	* @param float $x Static position
	* @return float
	*/
	function addMarginX($x) {
		$m = $this->getMargins();
		if ($this->getRTL()) {
			$x += $m["right"];
		} else {
			$x += $m["left"];
		}
		$this->SetX($x);
		return $x;
	}

	/**
	* Get the maximum line width to draw from the curren position -PGVRPDF
	* RTL supported
	* @return float
	*/
	function getMaxLineWidth() {
		$m = $this->getMargins();
		if ($this->getRTL()){
			return ($this->getRemainingWidth() + $m["right"]);
		} else {
			return ($this->getRemainingWidth() + $m["left"]);
		}
	}

	function getFootnotesHeight() {
		$h=0;
		foreach($this->printedfootnotes as $element) {
			$h+=$element->getHeight($this);
		}
		return $h;
	}

	/**
	* Returns the the current font size height -PGVRPDF
	* @return int
	*/
	function getCurrentStyleHeight() {
		if (empty($this->currentStyle)) {
			return $this->pgvreport->defaultFontSize;
		}
		$style = $this->pgvreport->getStyle($this->currentStyle);
		return $style["size"];
	}

	/**
	* Checks the Footnote and numbers them
	*
	* @param object &$footnote
	* @return boolean false if not numbered befor | object if already numbered
	*/
	function checkFootnote(&$footnote) {
		$ct = count($this->printedfootnotes);
		$val = $footnote->getValue();
		$i = 0;
		while($i < $ct) {
			if ($this->printedfootnotes[$i]->getValue() == $val) {
				// If this footnote already exist then set up the numbers for this object
				$footnote->setNum($i + 1);
				$footnote->setAddlink($i + 1);
				return $this->printedfootnotes[$i];
			}
			$i++;
		}
		// If this Footnote has not been set up yet
		$footnote->setNum($ct + 1);
		$footnote->setAddlink($this->AddLink());
		$this->printedfootnotes[] = $footnote;
		return false;
	}

	/**
	* Used this function instead of AddPage()
	* This function will make sure that images will not be overwritten
	*/
	function newPage() {
		if ($this->lastpicpage > $this->getPage()){
			$this->setPage($this->lastpicpage);
		}
		$this->AddPage();
	}


	/*******************************************
	* TCPDF protected functions
	*******************************************/

	/**
	* Add a page if needed -PGVRPDF
	* @param int $height Cell height. Default value: 0
	* @return boolean true in case of page break, false otherwise
	*/
	function checkPageBreakPDF($height) {
		return $this->checkPageBreak($height);
	}

	/**
	* Returns the remaining width between the current position and margins -PGVRPDF
	* @return float Remaining width
	*/
	function getRemainingWidthPDF() {
		return $this->getRemainingWidth();
	}

} //-- END PGVRPDF

/**
* Report Base object of PGVReportBase class inherited by PGVReportBasePDF
* @global PGVReportBasePDF $pgvreport
*/
$pgvreport = new PGVReportBasePDF();

$PGVReportRoot = $pgvreport;

/**
* Cell Element Class - PDF
*
* @see PGVRCell
* @see PGVRCellSHandler()
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVRCellPDF extends PGVRCell {
	/**
	* Create a Class CELL for PDF
	*
	* @param int $width Cell width (expressed in points)
	* @param int $height Cell height (expressed in points)
	* @param mixed $border Border Style
	* @param string $align Text alignement
	* @param string $bgcolor Background color code
	* @param string $style The name of the text style
	* @param int $ln Indicates where the current position should go after the call
	* @param mixed $top Y-position
	* @param mixed $left X-position
	* @param int $fill Indicates if the cell background must be painted (1) or transparent (0). Default value: 1
	* @param int $stretch Stretch carachter mode
	* @param string $bocolor Border color
	* @param string $tcolor Text color
	* @param boolean $reseth
	* @param boolean $padding
	*
	* @package PhpGedView
	* @subpackage Reports
	*/
	function __construct($width, $height, $border, $align, $bgcolor, $style, $ln, $top, $left, $fill, $stretch, $bocolor, $tcolor, $reseth, $padding) {
		parent::__construct($width, $height, $border, $align, $bgcolor, $style, $ln, $top, $left, $fill, $stretch, $bocolor, $tcolor, $reseth, $padding);
	}

	/**
	* PDF Cell renderer
	* @param PGVRPDF &$pdf
	*/
	function render(&$pdf) {
		/**
		* Use these variables to update/manipulate values
		* Repeated classes would update all their class variables again, Header/Page Header/Footer
		* This is the bugfree version
		*/
		$cX = 0;	// Class Left

		// Set up the text style
		if (($pdf->getCurrentStyle()) != ($this->styleName)) {
			$pdf->setCurrentStyle($this->styleName);
		}
		$tempText = str_replace(array("#PAGENUM#", "#PAGETOT#"), array($pdf->getAliasNumPage(), $pdf->getAliasNbPages()), $this->text);

		$match=array();
		// For Default @see PGVRCellSHandler()
		// HTML color to RGB => bgcolor, bocolor, tcolor
		if ($this->fill){
			preg_match("/#?(..)(..)(..)/", $this->bgcolor, $match);
			$r=hexdec($match[1]);
			$g=hexdec($match[2]);
			$b=hexdec($match[3]);
			$pdf->SetFillColor($r, $g, $b);
		}
		// Paint the Border color if set
		if (preg_match("/#?(..)(..)(..)/", $this->bocolor, $match)){
			$r=hexdec($match[1]);
			$g=hexdec($match[2]);
			$b=hexdec($match[3]);
			$pdf->SetDrawColor($r, $g, $b);
		}
		// Paint the text color or they might use inherited colors by the previous function
		if (preg_match("/#?(..)(..)(..)/", $this->tcolor, $match)){
			$r=hexdec($match[1]);
			$g=hexdec($match[2]);
			$b=hexdec($match[3]);
			$pdf->SetTextColor($r, $g, $b);
		}

		// If current position (left)
		if ($this->left == ".") {
			$cX = $pdf->GetX();
		}
		// For static position add margin (also updates X)
		else $cX = $pdf->addMarginX($this->left);

		// Check the width if set to page wide OR set by xml to larger then page wide
		if (($this->width == 0) or ($this->width > $pdf->getRemainingWidthPDF())) {
			$this->width = $pdf->getRemainingWidthPDF();
		}
		// For current position
		if ($this->top == ".") {
			$this->top = $pdf->GetY();
		} else {
			$pdf->SetY($this->top);
		}

		// Check the last cell height and adjust the current cell height if needed
		if ($pdf->lastCellHeight > $this->height) {
			$this->height = $pdf->lastCellHeight;
		}
		/**
		 * Check the Padding if set in XML file
		 * If set add 2 pt to all 4 sides
		 */
		$this->padding ? $pdf->SetCellPadding(2) : $pdf->SetCellPadding(0);

		// Check for pagebreak
		if (!empty($tempText)) {
			$cHT = $pdf->getNumLines($tempText, $this->width);
			$cHT = $cHT * $pdf->getCellHeightRatio() * $pdf->getCurrentStyleHeight();
			$cHT = $cHT;
			$cM = $pdf->getMargins();
			// Add padding
			if (!is_array($cM['cell'])) {
				$cHT += ($cM['cell'] * 2);
			} else {
				$cHT += ($cM['padding_top'] + $cM['padding_bottom']);
			}
			// Add a new page if needed
			if ($pdf->checkPageBreakPDF($cHT)) {
				$this->top = $pdf->GetY();
			}
			$tempText = stripControlCodes(spanLTRRTL($tempText, "BOTH"));
		}
		// HTML ready - last value is true
		$pdf->MultiCell($this->width, $this->height, $tempText, $this->border, $this->align, $this->fill, $this->newline, $cX, $this->top, $this->reseth, $this->stretch, true);
		// Reset the last cell height for the next line
		if ($this->newline >= 1) {
			$pdf->lastCellHeight = 0;
		}
		// OR save the last height if heigher then before
		elseif ($pdf->lastCellHeight < $pdf->getLastH()) {
			$pdf->lastCellHeight = $pdf->getLastH();
		}

		// Set up the url link if exists ontop of the cell
		if (!empty($this->url)) {
			$pdf->Link($cX, $this->top, $this->width, $this->height, $this->url);
		}
		// Reset the border and the text color to black or they will be inherited
		$pdf->SetDrawColor(0, 0, 0);
		$pdf->SetTextColor(0, 0, 0);
	}
}

/**
* HTML element - PDF Report
*
* @see PGVRHtml
* @package PhpGedView
* @subpackage Reports
*/
class PGVRHtmlPDF extends PGVRHtml {

	function __construct($tag, $attrs) {
		parent::__construct($tag, $attrs);
	}

	function render(&$pdf, $sub = false) {
		if (!empty($this->attrs["pgvrstyle"])) {
			$pdf->setCurrentStyle($this->attrs["pgvrstyle"]);
		}
		if (!empty($this->attrs["width"])) {
			$this->attrs["width"] *= 3.9;
		}

		$this->text = $this->getStart().$this->text;
		foreach($this->elements as $element) {
			if (is_string($element) && $element=="footnotetexts") {
				$pdf->Footnotes();
			} elseif (is_string($element) && $element=="addpage") {
				$pdf->newPage();
			} elseif ($element->get_type()=="PGVRHtml") {
				$this->text .= $element->render($pdf, true);
			} else {
				$element->render($pdf);
			}
		}
		$this->text .= $this->getEnd();
		if ($sub) {
			return $this->text;
		}
		$pdf->writeHTML($this->text); //prints 2 empty cells in the Expanded Relatives report
		return 0;
	}
}

/**
* TextBox element
*
* @see PGVRTextBox
* @package PhpGedView
* @subpackage Reports
*/
class PGVRTextBoxPDF extends PGVRTextBox {
	/**
	* Create a class Text Box for PDF
	*
	* @param float $width Text box width
	* @param float $height Text box height
	* @param string $border
	* @param string $bgcolor Background color code in HTML
	* @param boolean $newline
	* @param mixed $left
	* @param mixed $top
	* @param boolean $pagecheck=true
	* @param string $style
	* @param boolean $fill
	* @param boolean $padding=true
	* @param boolean $reseth Reset the last height after this bos is done
	*
	* @see PGVRTextBox
	* @package PhpGedView
	* @subpackage Reports
	*/
	function __construct($width, $height, $border, $bgcolor, $newline, $left, $top, $pagecheck, $style, $fill, $padding, $reseth) {
		parent::__construct($width, $height, $border, $bgcolor, $newline, $left, $top, $pagecheck, $style, $fill, $padding, $reseth);
	}

	/**
	* PDF Text Box renderer
	* @param PGVRPDF &$pdf
	*
	* @see PGVRTextBox
	* @see PGVRTextBoxPDF
	* @package PhpGedView
	* @subpackage Reports
	*/
	function render(&$pdf) {

		$newelements = array();
		$lastelement = "";
		$footnote_element = array();
		// Element counter
		$cE = count($this->elements);
		//-- collapse duplicate elements
		for($i = 0; $i < $cE; $i++){
			$element = $this->elements[$i];
			if (is_object($element)){
				if ($element->get_type() == "PGVRText"){
					if (!empty($footnote_element)){
						ksort($footnote_element);
						foreach ($footnote_element as $links){
							$newelements[] = $links;
						}
						$footnote_element = array();
					}
					if (empty($lastelement)){
						$lastelement = $element;
					} else {
						// Checking if the PGVRText has the same style
						if ($element->getStyleName() == $lastelement->getStyleName()){
							$lastelement->addText(str_replace("\n", "<br />", $element->getValue()));
						} elseif (!empty($lastelement)){
							$newelements[] = $lastelement;
							$lastelement = $element;
						}
					}
				}
				// Collect the Footnote links
				elseif ($element->get_type() == "PGVRFootnote"){
					// Check if the Footnote has been set with it's link number
					$pdf->checkFootnote($element);
					// Save first the last element if any
					if (!empty($lastelement)){
						$newelements[] = $lastelement;
						$lastelement = array();
					}
					// Save the Footnote with it's link number as key for sorting later
					$footnote_element[$element->num] = $element;
				}
				//-- do not keep empty footnotes
				elseif (($element->get_type() != "PGVRFootnote") || (trim($element->getValue()) != "")){
					if (!empty($footnote_element)){
						ksort($footnote_element);
						foreach ($footnote_element as $links){
							$newelements[] = $links;
						}
						$footnote_element = array();
					}
					if (!empty($lastelement)){
						$newelements[] = $lastelement;
						$lastelement = array();
					}
					$newelements[] = $element;
				}
			} else {
				if (!empty($lastelement)){
					$newelements[] = $lastelement;
					$lastelement = array();
				}
				if (!empty($footnote_element)){
					ksort($footnote_element);
					foreach ($footnote_element as $links){
						$newelements[] = $links;
					}
					$footnote_element = array();
				}
				$newelements[] = $element;
			}
		}
		if (!empty($lastelement)){
			$newelements[] = $lastelement;
		}
		if (!empty($footnote_element)){
			ksort($footnote_element);
			foreach ($footnote_element as $links){
				$newelements[] = $links;
			}
		}
		$this->elements = $newelements;
		unset($footnote_element, $lastelement, $links, $newelements);

		/**
		* Use these variables to update/manipulate values
		* Repeated classes would update all their class variables again, Header/Page Header/Footer
		* This is the bugfree version
		*/
		$cH = 0;	// Class Height
		$cW = 0;	// Class Width
		$cX = 0;	// Class Left
		$cY = 0;	// Class Top
		// Used with line breaks and TextBox height calculation within this box
		$pdf->largestFontHeight = 0;

		/**
		* Check the Padding if set in XML file
		* If set, add 2 pt to all 4 sides with TCPDF function SetCellPadding()
		*/
		$this->padding ? $pdf->SetCellPadding(2) : $pdf->SetCellPadding(0);

		// If current position (left)
		if ($this->left == ".") {
			$cX = $pdf->GetX();
		}
		// For static position add margin (returns and updates X)
		else $cX = $pdf->addMarginX($this->left);

		// If current position (top)
		if ($this->top == ".") {
			$cY = $pdf->GetY();
		} else {
			$cY = $this->top;
			$pdf->SetY($cY);
		}

		// Check the width if set to page wide OR set by xml to larger then page width (margin)
		if (($this->width == 0) or ($this->width > $pdf->getRemainingWidthPDF())) {
			$cW = $pdf->getRemainingWidthPDF();
		} else {
			$cW = $this->width;
		}

		// Save the original margins
		$cM = $pdf->getMargins();
		// Use cell padding to wrap the width
		// Temp Width with cell padding
		if (!is_array($cM['cell'])) {
			$cWT = $cW - ($cM['cell'] * 2);
		} else {
			$cWT = $cW - ($cM['padding_left'] + $cM['padding_right']);
		}
		// Element height (exept text)
		$eH = 0;
		$w = 0;
		// Temp Height
		$cHT = 0;
		/*
		* Text Info or Image Width
		*	0 => Last line width
		*	1 => 1 if text was wrapped, 0 if text did not wrap
		*	2 => number of LF
		* @var array $lw Array if Text or an Integer if Image
		*/
		$lw = array();
		// @var object $cE Element counter
		$cE = count($this->elements);
		//-- calculate the text box height + width
		for($i = 0; $i < $cE; $i++) {
			if (is_object($this->elements[$i])) {
				$ew = $this->elements[$i]->setWrapWidth($cWT - $w, $cWT);
				if ($ew == $cWT) {
					$w = 0;
				}
				$lw = $this->elements[$i]->getWidth($pdf);
				// Check if Array - An Integer with the Image Width, if not Text Array -> PHP 7.4 Error
				if (is_array($lw)) {
					// Text is already gets the # LF
					$cHT += $lw[2];
					if ($lw[1] == 1) {
						$w = $lw[0];
					} elseif ($lw[1] == 2) {
						$w = 0;
					} else {
						$w += $lw[0];
					}
					if ($w > $cW) {
						$w = $lw[0];
					}
				}
//	Footnote is at the bottom of the page. No need to calculate it's height or wrap the text!
//	We are changing the margins anyway!
				// For anything else but text (images), get the height
				$eH += $this->elements[$i]->getHeight($pdf);
			}
//			else {
//				$h += $pdf->getFootnotesHeight();
//			}
		}

		// Add up what's the final height
		$cH = $this->height;
		// If any element exist
		if ($cE > 0) {
			// Check if this is text or some other element, like images
			if ($eH == 0) {
				// This is text elements. Number of LF but at least one line
				$cHT = ($cHT+1) * $pdf->getCellHeightRatio();
				// Calculate the cell height with the largest font size used within this Box
				$cHT = $cHT * $pdf->largestFontHeight;
				// Add cell padding
				if ($this->padding) {
					if (!is_array($cM['cell'])) {
						$cHT += ($cM['cell'] * 2);
					} else {
						$cHT += ($cM['padding_top'] + $cM['padding_bottom']);
					}
					$cHT = $cHT;
				}
				if ($cH < $cHT) {
					$cH = $cHT;
				}
			}
			// This is any other element
			elseif ($cH < $eH) {
				$cH = $eH;
			}
		}
		$cH=ceil($cH);
		// Finally, check the last cell's height
		if ($cH < $pdf->lastCellHeight) {
			$cH = $pdf->lastCellHeight;
		}
		// Add a new page if needed
		if ($this->pagecheck) {
			// Reset last cell height or Header/Footer will inherit it, in case of pagebreak
			$pdf->lastCellHeight = 0;
			if ($pdf->checkPageBreakPDF($cH)) {
				$cY = $pdf->GetY();
			}
		}

		// Setup the border and background color
		$cS	= "";	// Class Style
		if ($this->border) $cS = "D";		// D or empty string: Draw (default)
		$match=array();
		// Fill the Background @see PGVRTextBoxSHandler()
		if ($this->fill){
			preg_match("/#?(..)(..)(..)/", $this->bgcolor, $match);
			$cS.="F";			// F: Fill the background
			$r=hexdec($match[1]);
			$g=hexdec($match[2]);
			$b=hexdec($match[3]);
			$pdf->SetFillColor($r, $g, $b);
		}
		// Draw the border
		if (!empty($cS)) {
			$pdf->Rect($cX, $cY, $cW, $cH, $cS);
		}
		// Add cell padding if set and if any text (element) exist
		if ($this->padding) {
			if ($cHT > 0) {
				if (!is_array($cM['cell'])) {
					$pdf->SetY($cY + $cM['cell']);
				}
			}
		}
		// Change the margins X, Width
		if (!$pdf->getRTL()) {
			if ($this->padding){
				if (!is_array($cM['cell'])) {
					$pdf->SetLeftMargin($cX + $cM['cell']);
				} else {
					$pdf->SetLeftMargin($cX + $cM['padding_left']);
				}
				$pdf->SetRightMargin($pdf->getRemainingWidthPDF() - $cW + $cM["right"]);
			}
			else {
				$pdf->SetLeftMargin($cX);
				$pdf->SetRightMargin($pdf->getRemainingWidthPDF() - $cW + $cM["right"]);
			}
		} else {
			if ($this->padding){
				if (!is_array($cM['cell'])) {
					$pdf->SetRightMargin($cX + $cM['cell']);
				} else {
					$pdf->SetRightMargin($cX + $cM['padding_right']);
				}
				$pdf->SetLeftMargin($pdf->getRemainingWidthPDF() - $cW  + $cM["left"]);
			} else {
				$pdf->SetRightMargin($cX);
				$pdf->SetLeftMargin($pdf->getRemainingWidthPDF() - $cW  + $cM["left"]);
			}
		}
		// Save the current page number
		$cPN = $pdf->getPage();

		// Render the elements (write text, print picture...)
		foreach($this->elements as $element) {
			if (is_object($element)) {
				$element->render($pdf);
			} elseif (is_string($element) and $element == "footnotetexts") {
				$pdf->Footnotes();
			} elseif (is_string($element) and $element == "addpage") {
				$pdf->newPage();
			}
		}
		// Restore the margins
		$pdf->SetLeftMargin($cM["left"]);
		$pdf->SetRightMargin($cM["right"]);

		// This will be mostly used to trick the multiple images last height
		if ($this->reseth) {
			$cH = 0;
			// This can only happen with multiple images and with pagebreak
			if ($cPN != $pdf->getPage()) {
				$pdf->setPage($cPN);
			}
		}
		// New line and some clean up
		if (!$this->newline) {
			$pdf->SetXY(($cX + $cW), $cY);
			$pdf->lastCellHeight = $cH;
		} else {
			// addMarginX() also updates X
			$pdf->addMarginX(0);
			$pdf->SetY($cY + $cH);
			$pdf->lastCellHeight = 0;
		}
		return true;
	}
}

/**
* Text Element Class
*
* @see PGVRText
* @package PhpGedView
* @subpackage Reports
*/
class PGVRTextPDF extends PGVRText {
	/**
	* Create a Text class for PDF
	*
	* @param string $style The name of the text style
	* @param string $color HTML color code
	*
	* @package PhpGedView
	* @subpackage Reports
	*/
	function __construct($style, $color) {
		parent::__construct($style, $color);
	}

	/**
	* PDF Text renderer
	* @param PGVRPDF &$pdf
	*
	* @package PhpGedView
	* @subpackage Reports
	*/
	function render(&$pdf) {
		// Set up the style
		if ($pdf->getCurrentStyle() != $this->styleName) {
			$pdf->setCurrentStyle($this->styleName);
		}
		$tempText = str_replace(array("#PAGENUM#", "#PAGETOT#"), array($pdf->getAliasNumPage(), $pdf->getAliasNbPages()), $this->text);

		// Paint the text color or they might use inherited colors by the previous function
		$match = array();
		if (preg_match("/#?(..)(..)(..)/", $this->color, $match)) {
			$r = hexdec($match[1]);
			$g = hexdec($match[2]);
			$b = hexdec($match[3]);
			$pdf->SetTextColor($r, $g, $b);
		} else {
			$pdf->SetTextColor(0, 0, 0);
		}

		if (substr_count($tempText, "\n") == 0) {
			// This text does not need special treatment
			$tempText = stripControlCodes(spanLTRRTL($tempText, "BOTH"));
			$pdf->writeHTML($tempText, false, false, true, false, "");
		} else {
			// Break text into separate lines, because TCPDF doesn't handle embedded NL very well
			$allLines = explode("\n", $tempText);
			$lineCount = count($allLines) - 1;
			for ($i=0; $i<$lineCount; $i++) {
				$tempText = stripControlCodes(spanLTRRTL($allLines[$i], "BOTH"));
				$tempText .= '<span><br /></span>';			// Make sure this is not a solitary <br /> -- (causes over-printing in TCPDF)
				$pdf->writeHTML($tempText, false, false, true, false, "");
			}
			$tempText = ltrim($allLines[$i], ' ');
			$tempText = stripControlCodes(spanLTRRTL($tempText, "BOTH"));
			$pdf->writeHTML($tempText, false, false, true, false, "");

			// Reset the text color to black or it will be inherited
			$pdf->SetTextColor(0, 0, 0);
		}
	}

	/**
	* Returns the height in points of the text element
	*
	* The height is already calculated in getWidth()
	* @param PGVRPDF &$pdf
	* @return float 0
	*/
	function getHeight(&$pdf) {
		return 0;
	}

	/**
	* Splits the text into lines if necessary to fit into a giving cell
	*
	* @param PGVRPDF &$pdf
	* @return array
	*/
	function getWidth(&$pdf) {
		// Setup the style name, a font must be selected to calculate the width
		if ($pdf->getCurrentStyle() != $this->styleName) {
			$pdf->setCurrentStyle($this->styleName);
		}
		// Check for the largest font size in the box
		$fsize = $pdf->getCurrentStyleHeight();
		if ($fsize > $pdf->largestFontHeight) {
			$pdf->largestFontHeight = $fsize;
		}

		// Get the line width
		$lw = $pdf->GetStringWidth($this->text);
		// Line Feed counter - Number of lines in the text
		$lfct = substr_count($this->text, "\n") + 1;
		// If there is still remaining wrap width...
		if ($this->wrapWidthRemaining > 0) {
			// Check with line counter too!
			// but floor the $wrapWidthRemaining first to keep it bugfree!
			$wrapWidthRemaining = floor($this->wrapWidthRemaining);
			if (($lw >= ($wrapWidthRemaining)) or ($lfct > 1)) {
				$newtext = "";
				$lines = explode("\n", $this->text);
				// Go throught the text line by line
				foreach($lines as $line) {
					// Line width in points + a little margin
					$lw = $pdf->GetStringWidth($line);
					// If the line has to be wraped
					if ($lw >= $wrapWidthRemaining) {
						$words = explode(" ", $line);
						$addspace = count($words);
						$lw = 0;
						foreach($words as $word) {
							$addspace--;
							$lw += $pdf->GetStringWidth($word." ");
							if ($lw <= $wrapWidthRemaining) {
								$newtext .= $word;
								if ($addspace != 0) {
									$newtext .= " ";
								}
							} else {
								$lw = $pdf->GetStringWidth($word." ");
								$newtext .= "\n$word";
								if ($addspace != 0) {
									$newtext .= " ";
								}
								// Reset the wrap width to the cell width
								$wrapWidthRemaining = $this->wrapWidthCell;
							}
						}
					} else {
						$newtext .= $line;
					}
					// Check the Line Feed counter
					if ($lfct > 1) {
						// Add a new line as long as it's not the last line
						$newtext.= "\n";
						// Reset the line width
						$lw = 0;
						// Reset the wrap width to the cell width
						$wrapWidthRemaining = $this->wrapWidthCell;
					}
					$lfct--;
				}
				$this->text = $newtext;
				$lfct = substr_count($this->text, "\n");
				return array($lw, 1, $lfct);
			}
		}
		$l = 0;
		$lfct = substr_count($this->text, "\n");
		if ($lfct > 0) {
			$l = 2;
		}
		return array($lw, $l, $lfct);
	}
}

/**
* Footnote element
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVRFootnotePDF extends PGVRFootnote {

	function __construct($style="") {
		parent::__construct($style);
	}

	/**
	* PDF Footnotes number renderer
	* Add Internal Link Number that will be used
	* to points to a reference in the Footer
	* @param PGVRPDF &$pdf
	*/
	function render(&$pdf) {
		$pdf->setCurrentStyle("footnotenum");
		/**
		* getCurrentStyleHeight() - From the user Style Input from XML (Font Height)
		* $this->numText - Add the Source number counter
		* $this->addlink - Internal counter for the Source Link that will point to the reference in the Footer
		* 					An Internal Link is a clickable area within the Report Document
		* @example BIRT:DATE [LINK-Number] => 2000-01-25 1
		* 		This Line :
		*			$pdf->Write($pdf->getCurrentStyleHeight(), "( ".$this->numText." )", $this->addlink);
		*		Will be printed like this with an Internal Link :
		*			2000-01-25 ( 1 )
		*
		* Write() is a TCPDF function that can handle URL or identifier returned by TCPDF function AddLink()
		*/
		$pdf->Write($pdf->getCurrentStyleHeight(), $this->numText, $this->addlink);
		/**
		* @deprecated 2021-01-20 - writeHTML() doesn't work with links
		* and the text is already <sup>
		* and it's messing up the font size, that becomes 'minimum'
		* insted of the chosed font size in the XML file
		*/
//		$pdf->writeHTML('<sup>'.$this->numText.'</sup>', false, false, false, false, "");
		// Why is this disabled ?
//		$this->wrapWidthRemaining -= 10.0;		// Account for the space taken by the footnote reference
	}

	/**
	* Write the Footnote text
	* Uses style name "footnote" by default
	*
	* @param PGVRPDF &$pdf
	*/
	function renderFootnote(&$pdf) {
		if ($pdf->getCurrentStyle() != $this->styleName) {
			$pdf->setCurrentStyle($this->styleName);
		}
		$tempText = str_replace(array("#PAGENUM#", "#PAGETOT#"), array($pdf->getAliasNumPage(), $pdf->getAliasNbPages()), $this->text);
		$tempText = str_replace(array('«', '»'), array('<u>', '</u>'), $tempText);		// Underline Source Title part of Source item
		// 
		/**
		* SetLink() is a TCPDF function.
		* The Link was created with TCPDF function AddLink()
		* and stored in $this->addlink to poin to a location within the document
		* Set the Link to this y/page position
		*/
		$pdf->SetLink($this->addlink, -1, -1);
		// Print first the Source Number Bold to stick out...
		if ($pdf->getRTL()) {
//			$pdf->writeHTML("<span> .".$this->num."</span>", false, false, false, false, ""); // @deprecated 2021-01-20 This doesn't work in PDF
			$pdf->writeHTML("<b> .".$this->num."</b>", false, false, false, false, "");
		} else {
//			$tempText = "<span>".$this->num.". </span>".$tempText; // @deprecated 2021-01-20 This doesn't work in PDF
			$pdf->writeHTML("<b>".$this->num.". </b>", false, false, false, false, "");
		}
		$tempText = stripControlCodes(spanLTRRTL($tempText, "BOTH"));
		// ... And then print the Source Text
		$pdf->writeHTML($tempText."<br />", true, false, true, false, "");
//		$pdf->Write($pdf->getCurrentStyleHeight(), $this->num.". ".$tempText."\n\n"); //@@ indi list of sources
	}

	/**
	* Returns the height in points of the Footnote element
	*
	* @param PGVRPDF &$pdf
	* @return float $h
	*/
	function getFootnoteHeight(&$pdf) {
//		$style = $pdf->getStyle($this->styleName);
//		$ct = substr_count($this->numText, "\n");
//		if ($ct > 0) {
//			$ct += 1;
//		}
//		$h = ($style["size"] * $ct);
//		return $h;
		return 0;
	}

	/**
	* Splits the text into lines to fit into a giving cell
	* and returns the last lines width
	*
	* @param PGVRPDF &$pdf
	* @return array
	*/
	function getWidth(&$pdf) {
		// Setup the style name, a font must be selected to calculate the width
		$pdf->setCurrentStyle("footnotenum");

		// Check for the largest font size in the box
		$fsize = $pdf->getCurrentStyleHeight();
		if ($fsize > $pdf->largestFontHeight) {
			$pdf->largestFontHeight = $fsize;
		}

		// Returns the Object if already numbered else false
		if (empty($this->num)){
			$pdf->checkFootnote($this);
		}

		// Get the line width
		$lw = ceil($pdf->GetStringWidth($this->numText));
		// Line Feed counter - Number of lines in the text
		$lfct = substr_count($this->numText, "\n") + 1;
		// If there is still remaining wrap width...
		if ($this->wrapWidthRemaining > 0) {
			// Check with line counter too!
			// but floor the $wrapWidthRemaining first to keep it bugfree!
			$wrapWidthRemaining = floor($this->wrapWidthRemaining);
			if (($lw >= $wrapWidthRemaining) or ($lfct > 1)) {
				$newtext = "";
				$lines = explode("\n", $this->numText);
				// Go through the text line by line
				foreach($lines as $line) {
					// Line width in points
					$lw = ceil($pdf->GetStringWidth($line));
					// If the line has to be wrapped
					if ($lw >= $wrapWidthRemaining) {
						$words = explode(" ", $line);
						$addspace = count($words);
						$lw = 0;
						foreach($words as $word) {
							$addspace--;
							$lw += ceil($pdf->GetStringWidth($word." "));
							if ($lw < $wrapWidthRemaining) {
								$newtext .= $word;
								if ($addspace != 0) {
									$newtext .= " ";
								}
							} else {
								$lw = $pdf->GetStringWidth($word." ");
								$newtext .= "\n$word";
								if ($addspace != 0) {
									$newtext .= " ";
								}
								// Reset the wrap width to the cell width
								$wrapWidthRemaining = $this->wrapWidthCell;
							}
						}
					} else {
						$newtext .= $line;
					}
					// Check the Line Feed counter
					if ($lfct > 1) {
						// Add a new line feed as long as it's not the last line
						$newtext.= "\n";
						// Reset the line width
						$lw = 0;
						// Reset the wrap width to the cell width
						$wrapWidthRemaining = $this->wrapWidthCell;
					}
					$lfct--;
				}
				$this->numText = $newtext;
				$lfct = substr_count($this->numText, "\n");
				return array($lw, 1, $lfct);
			}
		}
		$l = 0;
		$lfct = substr_count($this->numText, "\n");
		if ($lfct > 0) {
			$l = 2;
		}
		return array($lw, $l, $lfct);
	}
}

/**
* PageHeader element
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVRPageHeaderPDF extends PGVRPageHeader {

	function __construct() {
		parent::__construct();
	}

	/**
	* PageHeader element renderer
	* @param PGVRPDF &$pdf
	*/
	function render(&$pdf) {
		$pdf->clearPageHeader();
		foreach($this->elements as $element) {
			$pdf->addPageHeader($element);
		}
	}
}

/**
* PGVRImagePDF class element
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVRImagePDF extends PGVRImage {

	function __construct($file, $x, $y, $w, $h, $align, $ln) {
		parent::__construct($file, $x, $y, $w, $h, $align, $ln);
	}

	/**
	* PDF image renderer
	* @param PGVRPDF &$pdf
	*/
	function render(&$pdf) {
		global $lastpicbottom, $lastpicpage, $lastpicleft, $lastpicright;

		// Check for a pagebreak first
		if ($pdf->checkPageBreakPDF($this->height + 5)) {
			$this->y = $pdf->GetY();
		}

		$curx = $pdf->GetX();
		// If current position (left)set "."
		if ($this->x == ".") {
			$this->x = $pdf->GetX();
		}
		// For static position add margin
		else  {
			$this->x = $pdf->addMarginX($this->x);
			$pdf->SetX($curx);
		}
		if ($this->y == ".") {
			//-- first check for a collision with the last picture
			if (isset($lastpicbottom)) {
				if (($pdf->PageNo()==$lastpicpage) && ($lastpicbottom >= $pdf->GetY()) && ($this->x >= $lastpicleft) && ($this->x <= $lastpicright))
					$pdf->SetY($lastpicbottom + 5);
			}
			$this->y = $pdf->GetY();
		} else {
			$pdf->SetY($this->y);
		}

		$pdf->Image($this->file, $this->x, $this->y, $this->width, $this->height, "", "", $this->line, false, 72, $this->align);
		$lastpicpage = $pdf->PageNo();
		$pdf->lastpicpage = $pdf->getPage();
		$lastpicleft = $this->x;
		$lastpicright = $this->x + $this->width;
		$lastpicbottom = $this->y + $this->height;
		// Setup for the next line
		if ($this->line == "N") {
			$pdf->SetY($lastpicbottom);
		}
	}

	/**
	* Get the image height
	* @param PGVRPDF &$pdf
	* @return float
	*/
	function getHeight(&$pdf) {
		return $this->height;
	}

	function getWidth(&$pdf) {
		return $this->width;
	}
}

/**
* Line element
*
* @package PhpGedView
* @subpackage Reports
*/
class PGVRLinePDF extends PGVRLine {
	/**
	* Create a line class -PDF
	* @param mixed $x1
	* @param mixed $y1
	* @param mixed $x2
	* @param mixed $y2
	*/
	function __construct($x1, $y1, $x2, $y2) {
		parent::__construct($x1, $y1, $x2, $y2);
	}

	/**
	* PDF line renderer
	* @param PGVRPDF &$pdf
	*/
	function render(&$pdf) {
		if ($this->x1 == ".") $this->x1=$pdf->GetX();
		if ($this->y1 == ".") $this->y1=$pdf->GetY();
		if ($this->x2 == ".") {
			$this->x2 = $pdf->getMaxLineWidth();
		}
		if ($this->y2 == ".") $this->y2=$pdf->GetY();

		$pdf->Line($this->x1, $this->y1, $this->x2, $this->y2);
	}
}

/*
* Function stripControlCodes
*
*		This function strips UTF8 control codes from text before sending it to tcpdf
*
*		The following UTF8 control codes need to be removed when the currently active font does not support them:
*			PGV_UTF8_LRM  \xE2\x80\x8E  U+200E  (Left to Right mark:  zero-width character with LTR directionality)
*			PGV_UTF8_RLM  \xE2\x80\x8F  U+200F  (Right to Left mark:  zero-width character with RTL directionality)
*			PGV_UTF8_LRO  \xE2\x80\xAD  U+202D  (Left to Right override: force everything following to LTR mode)
*			PGV_UTF8_RLO  \xE2\x80\xAE  U+202E  (Right to Left override: force everything following to RTL mode)
*			PGV_UTF8_LRE  \xE2\x80\xAA  U+202A  (Left to Right embedding: treat everything following as LTR text)
*			PGV_UTF8_RLE  \xE2\x80\xAB  U+202B  (Right to Left embedding: treat everything following as RTL text)
*			PGV_UTF8_PDF  \xE2\x80\xAC  U+202C  (Pop directional formatting: restore state prior to last LRO, RLO, LRE, RLE)
*		See /includes/session.php
* @param string $text
*/
function stripControlCodes($text) {
	global $currentFont;
	// List of fonts that support UTF8 control codes.  Any fonts not in this list don't support those control codes.
	$fontUTF8 = array('dejavusans', 'dejavusanscondensed', 'dejavuserif', 'dejavuserifcondensed', 'freesans', 'freeserif');
	// List of UTF8 control codes that might need to be stripped from incoming text
	$controlUTF8 = array(PGV_UTF8_LRM, PGV_UTF8_RLM, PGV_UTF8_LRO, PGV_UTF8_RLO, PGV_UTF8_LRE, PGV_UTF8_RLE, PGV_UTF8_PDF);

	if (!in_array($currentFont, $fontUTF8)) {
		$text = str_replace($controlUTF8, '', $text);
	}

	return $text;
}

?>
