<?php
/**
 * Update the RA module database schema from version 0 to version 1
 *
 * Version 0: empty database
 * Version 1: create the tables, as per PGV 4.2.1
 *
 * The script should assume that it can be interrupted at
 * any point, and be able to continue by re-running the script.
 * Fatal errors, however, should be allowed to throw exceptions,
 * which will be caught by the framework.
 * It shouldn't do anything that might take more than a few
 * seconds, for systems with low timeout values.
 *
 * phpGedView: Genealogy Viewer
 * Copyright (C) 2009 Greg Roach
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @version $Id: db_schema_0_1.php 7377 2024-10-18 13:50:13Z canajun2eh $
 */

if (!defined('PGV_PHPGEDVIEW')) {
header('HTTP/1.0 403 Forbidden');
exit;
}

define('PGV_RA_DB_SCHEMA_0_1', '');

// Create all of the tables needed for this module

/* The tasks table holds individual tasks identified by t_id and stored in folder t_fr_id.
 * Tasks then have titles, descriptions, when they were created (t_startdate), when
 * they were completed (t_endate not NULL is completed) and assigned username
 * Two additional fields t_form and t_results were used previously to hold completed
 * task information linking a form type (t_form) and population of the form (t_results)
 */
if (!self::table_exists("{$TBLPREFIX}tasks")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}tasks (".
		" t_id          ".self::$INT4_TYPE."         NOT NULL,".
		" t_fr_id       ".self::$INT4_TYPE."         NOT NULL,".
		" t_title       ".self::$VARCHAR_TYPE."(255) NOT NULL,".
		" t_description ".self::$TEXT_TYPE."         NOT NULL,".
		" t_startdate   ".self::$INT4_TYPE."         NOT NULL,".
		" t_enddate     ".self::$INT4_TYPE."             NULL,".
		" t_results     ".self::$TEXT_TYPE."             NULL,".
		" t_form        ".self::$VARCHAR_TYPE."(255)     NULL,".
		" t_username    ".self::$VARCHAR_TYPE."(45)      NULL,".
		" PRIMARY KEY (t_id)".
		") ".self::$UTF8_TABLE
	);
} else {
	// t_form for the tasks table was created AFTER db_schem_0_0 (pre PGV 4.2.1)
	// so if upgrading from an earlier version we need to add the t_form field and initialize it
	if (!self::column_exists("{$TBLPREFIX}tasks", 't_form')) {
		self::exec("ALTER TABLE {$TBLPREFIX}tasks ADD t_form ".self::$VARCHAR_TYPE."(255) NULL");
	}
}

/* The comments table holds any comments related to a task and is identifed by c_id
 * Comments are only available to general tasks.  Source tasks
 * link back to general tasks so never have a direct link to this table.  Comments related to
 * individuals are stored in the user_comments as there is no link to a task
 * The c_t_id will be filled to link a comment with an individual task id from the tasks table
 * The c_u_username is filled with the username that added these comments so we can control
 * who can identify, modify or delete these comments in the future.
 * The c_body holds the comments themselves, while the c_datetime holds timestamp when this was created
*/
if (!self::table_exists("{$TBLPREFIX}comments")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}comments (".
		" c_id         ".self::$INT4_TYPE."        NOT NULL,".
		" c_t_id       ".self::$INT4_TYPE."        NOT NULL,".
		" c_u_username ".self::$VARCHAR_TYPE."(30) NOT NULL,".
		" c_body       ".self::$TEXT_TYPE."        NOT NULL,".
		" c_datetime   ".self::$INT4_TYPE."        NOT NULL,".
		" PRIMARY KEY (c_id)".
		") ".self::$UTF8_TABLE
	);
}

/* The tasksource table is used specifically to tie tasks to sources (rather than people)
 * The ts_t_id is the task id from the task table related to this source task while 
 * ts_s_id is the source id that this task is related to
 * ts_page is currently never updated and always NULL
 * ts_date is the date of creation - currently never updated and always NULL
 * ts_text is currently never updated and always NULL
 * ts_quay values are encoded values representing the quality of the data represented -  is currently never updated and always NULL
 *  (Unreliable = 1, Questionable = 2, Indirect = 4, Direct = 8, Secondary = 16, Primary = 32, Derivative = 64, Original = 128)
 *  so Questionable Primary and Indirect would be combined as 2+4+32 = 40 then encoded as Hexadecimal
 * ts_obje is currently never updated and always NULL
 * ts_array is currently never updated and always NULL
*/
if (!self::table_exists("{$TBLPREFIX}tasksource")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}tasksource (".
		" ts_t_id  ".self::$INT4_TYPE."         NOT NULL,".
		" ts_s_id  ".self::$VARCHAR_TYPE."(255) NOT NULL,".
		" ts_page  ".self::$VARCHAR_TYPE."(255)     NULL,".
		" ts_date  ".self::$VARCHAR_TYPE."(50)      NULL,".
		" ts_text  ".self::$TEXT_TYPE."             NULL,".
		" ts_quay  ".self::$VARCHAR_TYPE."(50)      NULL,".
		" ts_obje  ".self::$VARCHAR_TYPE."(20)      NULL,".
		" ts_array ".self::$TEXT_TYPE."             NULL,".
		" PRIMARY KEY (ts_s_id, ts_t_id)".
		") ".self::$UTF8_TABLE
	);
}

/* The folders table is an entity to collect tasks so each folder has fr_id.  Any task MUST have a folder id linked to it
 * The fr_name field is a convenient title/name to understand what the folder is used for
 * The fr_description field allows the user to add more detailed description of the folder use
 * The fr_parentid permits linking folders together making subfolders.  Root folders have fr_parentid = NULL
 */
if (!self::table_exists("{$TBLPREFIX}folders")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}folders (".
		" fr_id          ".self::$INT4_TYPE."         NOT NULL,".
		" fr_name        ".self::$VARCHAR_TYPE."(255)     NULL,".
		" fr_description ".self::$TEXT_TYPE."             NULL,".
		" fr_parentid    ".self::$INT4_TYPE."             NULL,".
		" PRIMARY KEY (fr_id)".
		") ".self::$UTF8_TABLE
	);
//	$statement=self::prepare("INSERT INTO {$TBLPREFIX}folders (fr_id, fr_name) VALUES (?, ?, ?, ?)");
//	// Do the insertion of Initial root folder so tasks can be created immediately
//	$statement->execute(array(1, 'Research Folder'));
}

/* The individualtask table is similar to the tasksource table and is used specifically to tie tasks to indi records (rather than sources)
 * The it_t_id is the task id from the task table related to this indi task while 
 * it_i_id is the indi id (ie Innn) that this task is related to.
 * So at any point you create a task and select a "Person" to link to that task then an entry is in this table linking the two
*/
if (!self::table_exists("{$TBLPREFIX}individualtask")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}individualtask (".
		" it_t_id   ".self::$INT4_TYPE."         NOT NULL,".
		" it_i_id   ".self::$VARCHAR_TYPE."(255) NOT NULL,".
		" it_i_file ".self::$INT4_TYPE."         NOT NULL,".
		" PRIMARY KEY (it_t_id, it_i_id,it_i_file)".
		") ".self::$UTF8_TABLE
	);
}

// This taskfacts table is currently not used
if (!self::table_exists("{$TBLPREFIX}taskfacts")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}taskfacts (".
		" tf_id       ".self::$INT4_TYPE."         NOT NULL,".
		" tf_t_id     ".self::$INT4_TYPE."             NULL,".
		" tf_factrec  ".self::$TEXT_TYPE."             NULL,".
		" tf_people   ".self::$VARCHAR_TYPE."(255)     NULL,".
		" tf_multiple ".self::$VARCHAR_TYPE."(3)       NULL,".
		" tf_type     ".self::$VARCHAR_TYPE."(4)       NULL,".
		" PRIMARY KEY (tf_id)".
		") ".self::$UTF8_TABLE
	);
} else {
	if (!self::column_exists("{$TBLPREFIX}taskfacts", 'tf_multiple')) {
		self::exec("ALTER TABLE {$TBLPREFIX}taskfacts ADD tf_multiple ".self::$VARCHAR_TYPE."(3) NULL");
	}
	if (!self::column_exists("{$TBLPREFIX}taskfacts", 'tf_type')) {
		self::exec("ALTER TABLE {$TBLPREFIX}taskfacts ADD tf_type ".self::$VARCHAR_TYPE."(4) NULL");
	}
}

/* The user comments table holds any comments related to a individual (ie Innn record) and is identifed by uc_id
 * User Comments do NOT have a associate task, but are linked to either people/individuals or families.  
 * The uc_username is filled with the username that added these comments so we can control
 * who can identify, modify or delete these comments in the future.
 * The uc_comment holds the comments themselves, while the uc_datetime holds timestamp when this was created
 * The uc_p_id will be filled to link a comment with an person/individual (Innn) reference
 * The uc_f_id will be filled with the PGV_GED_ID (the id of the PGV user logged in) - not sure why
*/
if (!self::table_exists("{$TBLPREFIX}user_comments")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}user_comments (".
		" uc_id       ".self::$INT4_TYPE."         NOT NULL,".
		" uc_username ".self::$VARCHAR_TYPE."(45)  NOT NULL,".
		" uc_datetime ".self::$INT4_TYPE."         NOT NULL,".
		" uc_comment  ".self::$TEXT_TYPE."         NOT NULL,".
		" uc_p_id     ".self::$VARCHAR_TYPE."(255) NOT NULL,".
		" uc_f_id     ".self::$INT4_TYPE."         NOT NULL,".
		" PRIMARY KEY (uc_id)".
		") ".self::$UTF8_TABLE
	);
}

/**
 * Note on the probabbilities table.
 * It is used to store the calculated percentages of specific facts.
 * An example of these facts follow.
 * The individuals surname matches their fathers 98% of the time
 *
 * The break down is the first level element, second level element, relationship, percentage
 */
if (!self::table_exists("{$TBLPREFIX}probabilities")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}probabilities (".
		" pr_id      ".self::$INT4_TYPE."         NOT NULL,".
		" pr_f_lvl   ".self::$VARCHAR_TYPE."(200) NOT NULL,".
		" pr_s_lvl   ".self::$VARCHAR_TYPE."(200) NOT NULL,".
		" pr_rel     ".self::$VARCHAR_TYPE."(200) NOT NULL,".
		" pr_matches ".self::$INT4_TYPE."         NOT NULL,".
		" pr_count   ".self::$INT4_TYPE."         NOT NULL,".
		" pr_file    ".self::$INT4_TYPE."             NULL,".
		" PRIMARY KEY (pr_id)".
		") ".self::$UTF8_TABLE
	);
}

/*
 * The factlookup table holds an array of facts (currently only Census and MILI facts)
 * that can be compared with the users lifetime to see if they should be flagged as missing info
 * Each entry has an id (auto incremented from previous), a description (this should be unique),
 * then a start and end date (each of the format yyyymmdd) that sets that date range to compare
 * a GEDCOM FACT this should be tied to (ie CENS or _MILI), then
 * pl_lv1, pl_lv2, pl_lv3, pl_lv4 and pl_lv5 represent place levels with pl_lv1 as country, pl_lv2 as state
 * pl_lv3 as county, pl_lv4 as city and pl_lv5 as neighbourhood.  Currently the code only checks
 * against pl_lv1 (country)
 * The sour_id and comment fields are currently not used
*/
if (!self::table_exists("{$TBLPREFIX}factlookup")) {
	self::exec(
		"CREATE TABLE {$TBLPREFIX}factlookup (".
		" id          ".self::$AUTO_ID_TYPE."      NOT NULL,".
		" description ".self::$VARCHAR_TYPE."(255) NOT NULL,".
		" startdate   ".self::$INT4_TYPE."         NOT NULL,".
		" enddate     ".self::$INT4_TYPE."         NOT NULL,".
		" gedcom_fact ".self::$VARCHAR_TYPE."(10)      NULL,".
		" pl_lv1      ".self::$VARCHAR_TYPE."(255)     NULL,".
		" pl_lv2      ".self::$VARCHAR_TYPE."(255)     NULL,".
		" pl_lv3      ".self::$VARCHAR_TYPE."(255)     NULL,".
		" pl_lv4      ".self::$VARCHAR_TYPE."(255)     NULL,".
		" pl_lv5      ".self::$VARCHAR_TYPE."(255)     NULL,".
		" sour_id     ".self::$VARCHAR_TYPE."(255)     NULL,".
		" comment     ".self::$VARCHAR_TYPE."(255)     NULL".
		") ".self::$UTF8_TABLE
	);
	$statement=self::prepare("INSERT INTO {$TBLPREFIX}factlookup (description, startdate, enddate, gedcom_fact, pl_lv1) VALUES (?, ?, ?, ?, ?)");
	// Do the insertion of Census facts at table instantiation
	// Since Census info is based on a specific day only put the range as that day, not the entire year
	$statement->execute(array('US Census 1800', 18000804, 18000804, 'CENS', 'USA')); // 4 August 1800
	$statement->execute(array('US Census 1810', 18100806, 18100806, 'CENS', 'USA')); // 6 August 1810
	$statement->execute(array('US Census 1820', 18200807, 18200807, 'CENS', 'USA')); // 7 August 1820
	$statement->execute(array('US Census 1830', 18300601, 18300601, 'CENS', 'USA')); // 1 June 1830
	$statement->execute(array('US Census 1840', 18400601, 18400601, 'CENS', 'USA')); // 1 June 1840
	$statement->execute(array('US Census 1850', 18500601, 18500601, 'CENS', 'USA')); // 1 June 1850
	$statement->execute(array('US Census 1860', 18600601, 18600601, 'CENS', 'USA')); // 1 June 1860
	$statement->execute(array('US Census 1870', 18700601, 18700601, 'CENS', 'USA')); // 1 June 1870
	$statement->execute(array('US Census 1880', 18800601, 18800601, 'CENS', 'USA')); // 1 June 1880
	$statement->execute(array('US Census 1890', 18900601, 18900601, 'CENS', 'USA')); // 1 June 1890
	$statement->execute(array('US Census 1900', 19000601, 19000601, 'CENS', 'USA')); // 1 June 1900
	$statement->execute(array('US Census 1910', 19100415, 19100415, 'CENS', 'USA')); // 15 April 1910
	$statement->execute(array('US Census 1920', 19200101, 19200101, 'CENS', 'USA')); // 1 January 1920
	$statement->execute(array('US Census 1930', 19300402, 19300402, 'CENS', 'USA')); // 2 April 1930
	$statement->execute(array('Canada Census 1871', 18710402, 18710402, 'CENS', 'Canada')); // 2 April 1871
	$statement->execute(array('Canada Census 1881', 18810404, 18810404, 'CENS', 'Canada')); // 4 April 1881
	$statement->execute(array('Canada Census 1891', 18910406, 18910406, 'CENS', 'Canada')); // 6 April 1891
	$statement->execute(array('Canada Census 1901', 19010331, 19010331, 'CENS', 'Canada')); // 31 March 1901
	$statement->execute(array('Canada Census 1911', 19110601, 19110601, 'CENS', 'Canada')); // 1 June 1911
	$statement->execute(array('Canada Census 1921', 19210601, 19210601, 'CENS', 'Canada')); // 1 June 1921
	$statement->execute(array('Canada Census 1931', 19310601, 19310601, 'CENS', 'Canada')); // 1 June 1931
	$statement->execute(array('UK Census 1841', 18410606, 18410606, 'CENS', 'UK' ));  // 6 June 1841
	$statement->execute(array('UK Census 1851', 18510330, 18510330, 'CENS', 'UK' )); // 30 March 1851
	$statement->execute(array('UK Census 1861', 18610407, 18610407, 'CENS', 'UK' )); // 7 April 1861
	$statement->execute(array('UK Census 1871', 18710402, 18710402, 'CENS', 'UK' )); // 2 April 1871
	$statement->execute(array('UK Census 1881', 18810403, 18810403, 'CENS', 'UK' )); // 3 April 1881
	$statement->execute(array('UK Census 1891', 18910405, 18910405, 'CENS', 'UK' )); // 5 April 1891
	$statement->execute(array('UK Census 1901', 19010331, 19010331, 'CENS', 'UK' )); // 31 March 1901
	$statement->execute(array('UK Census 1911', 19010402, 19010402, 'CENS', 'UK' )); // 2 April 1911
	$statement->execute(array('UK Census 1921', 19010619, 19010619, 'CENS', 'UK' )); // 19 June 1921

	// Insert War facts here
	$statement->execute(array('Civil War',  18610412, 18650409, '_MILI', 'USA')); // April 12, 1861 – April 9, 1865
	$statement->execute(array('WWI',        19140728, 19181111, '_MILI', null )); // July 28, 1914 – November 11, 1918
	$statement->execute(array('WWII',       19390901, 19450902, '_MILI', null )); // September 1, 1939 – September 2, 1945
	$statement->execute(array('Korean War', 19500625, 19530727, '_MILI', null )); // June 25, 1950 – July 27, 1953
	$statement->execute(array('Vietnam War', 19551101, 19750430, '_MILI', null )); // November 1, 1955 – April 30, 1975
}

// Update the version to indicate sucess
set_site_setting($schema_name, $next_version);
