<?php

/**
 * phpGedView Research Assistant Tool - Functions File.
 *
 * phpGedView: Genealogy Viewer
 * Copyright (C) 2002 to 2024 PGV Development Team.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @package PhpGedView
 * @subpackage Research_Assistant
 * @version $Id: ra_functions.php 7377 2024-10-18 13:50:13Z canajun2eh $
 * @author Jason Porter
 * @author Wade Lasson
 * @author Brandon Gagnon
 * @author Brian Kramer
 * @author Julian Gautier
 * @author Mike Hessick
 * @author Mike Austin
 * @author Gavin Winkler
 * @author David Molton
 * @author Daniel Parker
 */

if (!defined('PGV_PHPGEDVIEW')) {
	header('HTTP/1.0 403 Forbidden');
	exit;
}

loadLangFile("research_assistant:lang, research_assistant:help_text");

require_once PGV_ROOT.'modules/research_assistant/forms/ra_privacy.php';
require_once PGV_ROOT.'modules/research_assistant/forms/ra_RSFunction.php';
require_once PGV_ROOT.'modules/research_assistant/forms/ra_RSSingleFactClass.php';

if (file_exists($INDEX_DIRECTORY.$GEDCOM.'_ra_priv.php')) {
	require_once $INDEX_DIRECTORY.$GEDCOM.'_ra_priv.php';
}
define("BASEPATH", './modules/research_assistant/');

/**
 * Base class for the Research Assistant, contains all basic functionality
 */
class ra_functions {

	var $sites = array();
	function Init() {
		global $TBLPREFIX;
		try {
			PGV_DB::updateSchema('modules/research_assistant/db_schema/', 'RA_SCHEMA_VERSION', 1);
		} catch (PDOException $ex) {
			// The schema update scripts should never fail.  If they do, there is no clean recovery.
			die($ex);
		}
		
		// If the factlookup table has not been initialized with all the Canada CENS info then clear the existig table and prepare it again
		// If the db_schema file changes to add or change additional facts in the future then the next code will need change and the instantiation
		// details from the db_schema files should be copied below.  Older pre 4.3.1 SVN 7376 only had 25 rows without Canada CENS info
		// and had a mistake in US 1840 census, so easiest way is to check for less than 26 lines and dump and reload
		$count = PGV_DB::prepare("SELECT id FROM {$TBLPREFIX}factlookup")
			->fetchAll();
		if (count($count) <= 25) {
			PGV_DB::prepare("DELETE FROM {$TBLPREFIX}factlookup")
			->execute();
			$statement=PGV_DB::prepare("INSERT INTO {$TBLPREFIX}factlookup (description, startdate, enddate, gedcom_fact, pl_lv1) VALUES (?, ?, ?, ?, ?)");
			// Do the insertion of Census facts at table instantiation
			// Since Census info is based on a specific day only put the range as that day, not the entire year
			$statement->execute(array('US Census 1800', 18000804, 18000804, 'CENS', 'USA')); // 4 August 1800
			$statement->execute(array('US Census 1810', 18100806, 18100806, 'CENS', 'USA')); // 6 August 1810
			$statement->execute(array('US Census 1820', 18200807, 18200807, 'CENS', 'USA')); // 7 August 1820
			$statement->execute(array('US Census 1830', 18300601, 18300601, 'CENS', 'USA')); // 1 June 1830
			$statement->execute(array('US Census 1840', 18400601, 18400601, 'CENS', 'USA')); // 1 June 1840
			$statement->execute(array('US Census 1850', 18500601, 18500601, 'CENS', 'USA')); // 1 June 1850
			$statement->execute(array('US Census 1860', 18600601, 18600601, 'CENS', 'USA')); // 1 June 1860
			$statement->execute(array('US Census 1870', 18700601, 18700601, 'CENS', 'USA')); // 1 June 1870
			$statement->execute(array('US Census 1880', 18800601, 18800601, 'CENS', 'USA')); // 1 June 1880
			$statement->execute(array('US Census 1890', 18900601, 18900601, 'CENS', 'USA')); // 1 June 1890
			$statement->execute(array('US Census 1900', 19000601, 19000601, 'CENS', 'USA')); // 1 June 1900
			$statement->execute(array('US Census 1910', 19100415, 19100415, 'CENS', 'USA')); // 15 April 1910
			$statement->execute(array('US Census 1920', 19200101, 19200101, 'CENS', 'USA')); // 1 January 1920
			$statement->execute(array('US Census 1930', 19300402, 19300402, 'CENS', 'USA')); // 2 April 1930
			$statement->execute(array('Canada Census 1871', 18710402, 18710402, 'CENS', 'Canada')); // 2 April 1871
			$statement->execute(array('Canada Census 1881', 18810404, 18810404, 'CENS', 'Canada')); // 4 April 1881
			$statement->execute(array('Canada Census 1891', 18910406, 18910406, 'CENS', 'Canada')); // 6 April 1891
			$statement->execute(array('Canada Census 1901', 19010331, 19010331, 'CENS', 'Canada')); // 31 March 1901
			$statement->execute(array('Canada Census 1911', 19110601, 19110601, 'CENS', 'Canada')); // 1 June 1911
			$statement->execute(array('Canada Census 1921', 19210601, 19210601, 'CENS', 'Canada')); // 1 June 1921
			$statement->execute(array('Canada Census 1931', 19310601, 19310601, 'CENS', 'Canada')); // 1 June 1931
			$statement->execute(array('UK Census 1841', 18410606, 18410606, 'CENS', 'UK' ));  // 6 June 1841
			$statement->execute(array('UK Census 1851', 18510330, 18510330, 'CENS', 'UK' )); // 30 March 1851
			$statement->execute(array('UK Census 1861', 18610407, 18610407, 'CENS', 'UK' )); // 7 April 1861
			$statement->execute(array('UK Census 1871', 18710402, 18710402, 'CENS', 'UK' )); // 2 April 1871
			$statement->execute(array('UK Census 1881', 18810403, 18810403, 'CENS', 'UK' )); // 3 April 1881
			$statement->execute(array('UK Census 1891', 18910405, 18910405, 'CENS', 'UK' )); // 5 April 1891
			$statement->execute(array('UK Census 1901', 19010331, 19010331, 'CENS', 'UK' )); // 31 March 1901
			$statement->execute(array('UK Census 1911', 19010402, 19010402, 'CENS', 'UK' )); // 2 April 1911
			$statement->execute(array('UK Census 1921', 19010619, 19010619, 'CENS', 'UK' )); // 19 June 1921

			// Insert War facts here
			$statement->execute(array('Civil War',  18610412, 18650409, '_MILI', 'USA')); // April 12, 1861 – April 9, 1865
			$statement->execute(array('WWI',        19140728, 19181111, '_MILI', null )); // July 28, 1914 – November 11, 1918
			$statement->execute(array('WWII',       19390901, 19450902, '_MILI', null )); // September 1, 1939 – September 2, 1945
			$statement->execute(array('Korean War', 19500625, 19530727, '_MILI', null )); // June 25, 1950 – July 27, 1953
			$statement->execute(array('Vietnam War', 19551101, 19750430, '_MILI', null )); // November 1, 1955 – April 30, 1975
		}
	}

	/*
	 * Gets events from within the date range supplied
	 *
	 * @startDate the Starting date you want to look from, in the format yyyymmdd
	 * @endDate the Ending date for the range you want to look, in the format yyyymmdd
	 * @factLookingFor optional fact to narrow to a specific GEDCOM facttype
	 * @place the place you want to narrow your search to.  Comma delimited. I.E(USA,IDAHO,BOISE) Up to 5 total criteria
	 *
	 * @return A multi-dimensional array of the valid dates
	 */
	function getEventsForDates($startDate,$endDate,$factLookingFor = "",$place = "") {
		global $TBLPREFIX;

		if (empty($endDate)) {
			//Add a ten year difference if no end date was sent in
			$endDate = $startDate + 100000;
		}

		if (empty($factLookingFor)) {
			$sql = "SELECT * FROM {$TBLPREFIX}factlookup WHERE startdate<=? AND enddate>=?";
			$vars=array($endDate, $startDate);
		} else {
			$sql = "SELECT * FROM {$TBLPREFIX}factlookup WHERE startdate<=? AND enddate>=? AND gedcom_fact ".PGV_DB::$LIKE." ?";
			$vars=array($endDate, $startDate, "%{$factLookingFor}%");
		}

		if (!empty($place)) {
			$parts = explode(',', $place);
			for ($i=0; $i<count($parts); $i++) {
				$parts[$i]=trim($parts[$i]);
			}

			if (count($parts) > 0) {
				$numOfParts=count($parts) -1;

// Since all the factlookup are currently only based on country then we have to ignore all but LVL1 place data
				$sql.=" AND pl_lv1 ".PGV_DB::$LIKE." ?";
				$vars[]='%'.$parts[$numOfParts].'%';

/* This code exists for the future if have to support more than LVL1 place data
				for ($i=0; ($i<count($parts) && $i<5); $i++) {
					if (!empty($parts[$numOfParts])) {
						$sql.=" AND pl_lv".($i+1)." ".PGV_DB::$LIKE." ?";
						$vars[]='%'.$parts[$numOfParts].'%';
					}
					$numOfParts--;
				}
*/
			}
		} else {
			$sql.=" AND PL_LV1 IS NULL";
		}
		return
			PGV_DB::prepare($sql)
			->execute($vars)
			->fetchAll(PDO::FETCH_ASSOC);
	}

	/**
	 * gets the details for a task from the database
	 * @param int taskid
	 * @return array
	 */
	function getTask($taskid) {
		global $TBLPREFIX;

		return
			PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_id=?")
			->execute(array($taskid))
			->fetchOneRow(PDO::FETCH_ASSOC);
	}

	/**
	 * Displays the menu for the research_assistant module
	 *
	 * This function will print the main menu for the research_assistant.
	 * It will also do checks to make sure they are in a folder and display the relevant
	 * menu assications to what the user is doing. That way you cant delete a folder if you aren't in one etc..
	 *
	 * @param string $folderid
	 * @access public
	 * @return mixed
	 */
	function print_menu($folderid = "", $taskid = "") {
		// Grab the global language array for internationalization
		global $pgv_lang, $TBLPREFIX;

		$width = 150;

		if (empty ($folderid) && !empty ($taskid)) {
			$folderid=
				PGV_DB::prepare("SELECT t_fr_id FROM {$TBLPREFIX}tasks WHERE t_id=?")
				->execute(array($taskid))
				->fetchOne();
		}

		// Restrict the column if we are not at the top of the module
		if ($folderid == "") {
			$percent = "";
		} else {
			$percent = " width=\"22%\" ";
		}

		// Display for the menu
		global $SHOW_MY_TASKS, $SHOW_ADD_TASK, $SHOW_AUTO_GEN_TASK, $SHOW_VIEW_FOLDERS;
		global $SHOW_ADD_FOLDER, $SHOW_ADD_UNLINKED_SOURCE, $SHOW_VIEW_PROBABILITIES;//show

		$out = '<table class="list_table" width="100%" cellpadding="2">';
		$out .= '<tr>';
		$out .= '<td align="left"'.$percent.'class="optionbox wrap">'.ra_functions :: print_top_folder($folderid).'</td>';
		$out .= '<td align="center" class="optionbox" width="'.$width.'"><a href="module.php?mod=research_assistant">'.
			'<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
			$pgv_lang["research_assistant"].'" border="0"></img><br />'.$pgv_lang["research_assistant"].'</a></td>';

		//button 'My Tasks'
		if (PGV_USER_ACCESS_LEVEL<=$SHOW_MY_TASKS)
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.
				'<a href="module.php?mod=research_assistant&amp;action=mytasks">'.
				'<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
				$pgv_lang["ra_my_tasks"].'" border="0"></img><br />'.$pgv_lang["ra_my_tasks"].'</a></td>';

		//button 'Add Task''
		if (PGV_USER_ACCESS_LEVEL<=$SHOW_ADD_TASK)
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.
				'<a href="module.php?mod=research_assistant&amp;action=addtask&amp;folderid='.$folderid.'">'.
				'<img src="modules/research_assistant/images/add_task.gif" alt="'.$pgv_lang["ra_add_task"].
				'" border="0"></img><br />'.$pgv_lang["ra_add_task"].'</a></td>';

		//button 'View Folders'
		if (PGV_USER_ACCESS_LEVEL<=$SHOW_VIEW_FOLDERS)
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.
				'<a href="module.php?mod=research_assistant&amp;action=view_folders">'.
				'<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
				$pgv_lang["ra_view_folders"].'" border="0"></img><br />'.$pgv_lang["ra_view_folders"].'</a></td>';

		//button 'Add Folder'
		if (PGV_USER_ACCESS_LEVEL<=$SHOW_ADD_FOLDER )
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.
				'<a href="module.php?mod=research_assistant&amp;action=addfolder">'.
				'<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
				$pgv_lang["ra_add_folder"].'" border="0"></img><br />'.
				$pgv_lang["ra_add_folder"].'</a></td>';

		// Below here is "in folder" relevant information. These are only shown when the user is inside a folder.
		if (!empty ($folderid)) {
			$folderparent=
				PGV_DB::prepare("SELECT fr_parentid FROM {$TBLPREFIX}folders WHERE fr_id=?")
				->execute(array($folderid))
				->fetchOne();

			// Print folder parent link
			if (!empty ($folderparent))
				$url = '<a href="module.php?mod=research_assistant&amp;action=viewtasks&amp;folderid='.$folderparent.'">';
			else
				$url = '<a href="module.php?mod=research_assistant&amp;action=view_folders">';

			// Finish up the links
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.$url.
				'<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
				$pgv_lang["ra_up_folder"].'" border="0"></img><br />'.$pgv_lang["ra_up_folder"].'</a></td>';
			$out .= '<td align="center" class="optionbox" width="'.$width.'">'.
				'<a href="module.php?mod=research_assistant&amp;action=editfolder&amp;folderid='.
				$folderid.'"><img src="modules/research_assistant/images/folder_blue_icon.gif" alt="'.
				$pgv_lang["ra_edit_folder"].'" border="0" /><br />'.$pgv_lang["ra_edit_folder"].'</a></td>';
		}

		// Close up
		$out .= '<td align="right" class="optionbox">  </td>';
		$out .= '</tr>';
		//$out .= '</table>';

		return $out;
	}

	/**
	 * Status message display
	 *
	 * This function prints a lovely formatted confirmation or error message depending on paramaters passed int
	 *
	 * @param mixed $message The message you want to display.
	 * @param bool $status true, false if the message is success or failure. (This controls the image shown either a check mark, or an X)
	 * @return mixed
	 */
	function printMessage($message, $status) {
		// Setup which kind of message it is
		if ($status) {
			$status = "Success!";
			$output="<br/>";

			//gets each person that was changed and shows their name with a link to their individual page
			if (isset($_REQUEST['personid'])) {
				$thePeopleList=explode(";",$_REQUEST['personid']);
				foreach($thePeopleList as $i=>$pid) {
					if(!empty($pid)) {
						$person=Person::getInstance($pid);
						$output.='<a href="'.$person->getLinkUrl().'">'.$person->getFullName().'</a><br/>';
					}
				}
			}

			$output.="<br/>";
			$div = "<span class=\"warning\">";
			$end = "</span>";
		} else {
			$status = "Error!";
			$image = "modules/research_assistant/images/xbutton.gif";
			$div = "<div class=\"error\">";
			$end = "</div>";
		}

		// The actual message layout
		$out = '<table align="center" width="50%" height="100" valign="center">';
		if (!empty($image)) $out .= '<tr><td class="optionbox" align="center" valign="center"><img src="'.$image.'" /></td></tr>';
		$out .= '<tr><td class="optionbox" valign="center" align="center"><h3>'.$div.''.$message.''.$end.'</h3></td></tr>';
		$out .= '</table>';

		return $out;
	}

	/**
	 * Prints the edit_Folder info to the user
	 *
	 * @param mixed $folder_id
	 * @return mixed
	 */
	function edit_Folder($folder_id) {
		// Get the information from the form and display it to the user
		$editfrm = $this->print_form('ra_EditFolder');

		return $editfrm->content($folder_id);
	}

	/**
	* This is the short Description for the Function
	*
	* @param string $name Name of the form to display
	* @return string HTML of the form
	*/
	function print_form($name = '') {
		$out = "";
		$ext = "";
		// Make sure we have something to display
		if (!empty ($name)) {

			// Add the php extension if it's not there
			if (strstr($name, ".php") === false)
				$ext = ".php";

			$path = BASEPATH."/forms/".$name.$ext;

			// Display RA submenu plus other tools or show an error
			if (file_exists($path)) {
				// Load the form.
				require_once $path;
				$form = new $name ();
				$out = $form->display_form();
				return $out;
			} else {
				return '<div class="error">ERROR: File '.BASEPATH.'forms/'.$name.'.php was not found.</div>';
			}
		}

		// Show error
		if (empty ($out))
			$out = '<div class="error">ERROR: There was a problem loading a form, or no form was specified.</div>';

		return $out;
	}

	/**
	 * Basic print_form function that uses the buffer approach
	 *
	 * @param mixed $name
	 * @return mixed
	 */
	function print_simple_form($name) {
		$path = BASEPATH."forms/".$name.".php";

		// Find and print the form otherwise display an error
		if (file_exists($path)) {
			// Load the form.
			return $this->get_include_contents($path);
		} else {
			return "<div class=\"error\">ERROR: File modules/research_assistant/forms/".$name.".php was not found.</div>";
		}
	}

	/**
	 * Gets the contents of a file to display
	 *
	 * Uses the buffer approach to parse a form and return that via a variable
	 * @param mixed $filename
	 * @return mixed
	 */
	function get_include_contents($filename) {
		// Print the form if it exists
		if (is_file($filename)) {
			ob_start();
			require $filename;
			$contents = ob_get_contents();
			ob_end_clean();

			return $contents;
		}
		return false;
	}

	/**
	 * Pulls all of the task data from the database
	 *
	 * @param mixed $folderId folderId to search for
	 * @param String $orderby If passed, query the database to order by
	 * @return mixed Results from the database
	 */
	function get_folder_list($folderId, $orderby = "") {
		// Obtain the table prefix from the site
		global $TBLPREFIX;

		if (!empty ($orderby)) {
			// Switch order by
			$this->switch_order();

			$sql="SELECT * FROM {$TBLPREFIX}tasks WHERE t_fr_id=? ORDER BY ".$orderby." ".$_REQUEST["type"];
		} else {
			$sql="SELECT * FROM {$TBLPREFIX}tasks WHERE t_fr_id=?";
		}

		return
			PGV_DB::prepare($sql)
			->execute(array($folderId))
			->fetchAll();
	}

	/**
	 * GETS all available FOLDERS and creates a combo box with the folders listed.
	 *
	 * @return all available folders
	 */

	function getFolders() {
		global $TBLPREFIX;

		$rows=
			PGV_DB::prepare("select fr_name, fr_id from {$TBLPREFIX}folders")
			->fetchAll();

		$out = "";
		foreach ($rows as $row) {
			$out .= '<option value="'.$row->fr_id.'"';
			if (!empty($_REQUEST['folderid']) && $_REQUEST['folderid']==$row->fr_id) {
				$out .= '" selected="selected"';
			}
			$out .= '>'.$row->fr_name.'</option>';
		}
		return $out;
	}

	/**
	 * Gets a list of assigned users tasks
	 *
	 *
	 */
	function get_user_tasks($userName) {
		global $TBLPREFIX;
		return
			PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_username=? AND t_enddate IS NULL")
			->execute(array($userName))
			->fetchAll();
	}

	/**
	 * Displays a list of folders and their tasks
	 *
	 * @param mixed $folderId FolderId in the Database
	 * @return string HTML code for the tasks
	 */
	function print_list($folderId, $orderId = "") {
		// Set a defualt order by
		if (!isset ($_REQUEST["type"])) {
			$_REQUEST["type"] = "asc";
		}
		$rows = $this->get_folder_list($folderId, $orderId);
		$out = $this->print_tasks($folderId, $rows);
		if (empty ($out)) {
			$out = "<div class=\"error\">ERROR: Nothing was found in your database.</div>";
		}
		return $out;
	}

	/*
	 * Display tasks that a user has assigned to them.
	 */
	function print_user_list($userName)
	{
		if(!empty($_REQUEST["Filter"]))
			$filter = $_REQUEST["Filter"];
		else $filter = "Incomplete";

		global $TBLPREFIX, $pgv_lang;
		switch ($filter) {
		case 'Completed':
			$statement=PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_username=? AND t_enddate IS NOT NULL");
			break;
		case 'Incomplete':
			$statement=PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_username=? AND t_enddate IS NULL");
			break;
		case 'All':
		default:
			$statement=PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_username=?");
			break;
		}
		$rows=$statement->execute(array($userName))->fetchAll();

		// Display the Tasks table header
		$out  = '<table id="Tasks" class="list_table" align="center" width="700" border="0">';
		$out .= '<thead><tr><th colspan="7" class="topbottombar"><h2>'.
			print_help_link("ra_view_tasklist_help", "qm", '', false, true).$pgv_lang["ra_Task_View"].'</h2>';

		$mytasksfrmAction = "module.php?mod=research_assistant&amp;action=mytasks";
		$out .= '<form name="mytasks" method="GET" action="'.$mytasksfrmAction.'">'."\n";
		$out .= '<input type="hidden" name="mod" value="research_assistant" />'."\n";
		$out .= '<input type="hidden" name="action" value="mytasks" />'."\n";

		// Display a drop down filter for selecting different Task states
		$out .= '<p>'.$pgv_lang["ra_FilterBy"].': <select name="Filter" onchange="document.mytasks.submit()">';
		$out .= '<option ';
		if ($filter == "All") $out .= 'selected="selected" ';
		$out .= 'value="All">'.$pgv_lang["ra_All"].'</option>';
		$out .= '<option ';
		if ($filter == "Completed") $out .= 'selected="selected" ';
		$out .= 'value="Completed">'.$pgv_lang["ra_completed"].'</option>';
		$out .= '<option ';
		if ($filter == "Incomplete") $out .= 'selected="selected" ';
		$out .= 'value="Incomplete">'.$pgv_lang["ra_incomplete"].'</option>';
		$out .= '</select></form></th></tr>';

		// Display the table column headers
		$out .= '<tr><th class="list_label"><a href="module.php?mod=research_assistant&amp;action=mytasks&amp;Filter='.
			$filter.'&amp;folderid=&amp;orderby=t_title&amp;type=">'.$pgv_lang["ra_title"].'</a></th>'.
			'<th class="list_label"><a href="module.php?mod=research_assistant&amp;action=mytasks&amp;Filter='.
			$filter.'&amp;folderid=&amp;orderby=t_description&amp;type=">'.$pgv_lang["ra_description"].'</a></th>'.
			'<th class="list_label"><a href="module.php?mod=research_assistant&amp;action=mytasks&amp;Filter='.
			$filter.'&amp;folderid=&amp;orderby=t_startdate&amp;type=">'.$pgv_lang["ra_Start_Date"].'</a></th>'.
			'<th class="list_label"><a href="module.php?mod=research_assistant&amp;action=mytasks&amp;Filter='.
			$filter.'&amp;folderid=&amp;orderby=t_enddate&amp;type=">'.$pgv_lang["ra_completed"].'</a></th>'.
			'<th class="list_label">'.$pgv_lang["ra_edit"].'</th><th class="list_label">'.$pgv_lang["ra_delete"]."</th>\n".
			'<th class="list_label">'.$pgv_lang["ra_complete"].'</tr></thead>';

		// Display each of the rows
		foreach ($rows as $row) {
			if (empty ($row->t_enddate)) {
				$completeLink = '<a href="module.php?mod=research_assistant&amp;action=completeTask'.
						'&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_complete"].'</a>';
				$date=timestamp_to_gedcom_date($row->t_startdate);
				$out .= '<tr><td class="list_value"><a href="module.php?mod=research_assistant&amp;'.
					'action=viewtask&amp;taskid='.$row->t_id.'" class="list_item name2">'.PrintReady($row->t_title).'</a></td>'.
					'<td class="list_value_wrap">'.($row->t_description).'</td>'.
					'<td class="list_value" align="center">'.$date->Display(false).'</td>'.
					'<td class="list_value" align="center">'.$this->checkComplete($row).'</td>';

				if(PGV_USER_IS_ADMIN || PGV_USER_NAME==($row->t_username)){
				// If owner of the row info or an admin then display the edit and delete task links

				$out .= '<td class="list_value" align="center"><a href="module.php?mod=research_assistant'.
					'&amp;action=edittask&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_edit"].'</a></td>'.
					'<td class="list_value" align="center"><a href="module.php?mod=research_assistant'.
					'&amp;action=deletetask&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_delete"].'</a></td>'.
					'<td class="list_value" align="center">'.$completeLink.'</td>';
				} else {
					$out .= '<td class="list_value" align="center">&nbsp;</td>'.
						'<td class="list_value" align="center">&nbsp;</td>'.
						'<td class="list_value" align="center">&nbsp;</td>';
				}
				$out .= '</tr>';
			}
			else
			{
				// while we set completeLink, because forms are currently disabled we temporarily remove the printing of this edit form data URL
				$completeLink = '<a href="module.php?mod=research_assistant&amp;action=completeTask'.
						'&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_editform"].'</a>';
				$date=timestamp_to_gedcom_date($row->t_startdate);
				$out .= '<tr><td class="list_value"><a href="module.php?mod=research_assistant&amp;'.
					'action=viewtask&amp;taskid='.$row->t_id.'" class="list_item name2">'.PrintReady($row->t_title).'</a></td>'.
					'<td class="list_value_wrap">'.$row->t_description.'</td>'.
					'<td class="list_value" align="center">'.$date->Display(false).'</td>'.
					'<td class="list_value" align="center">'.$this->checkComplete($row).'</td>';

				if(PGV_USER_IS_ADMIN || PGV_USER_NAME==($row->t_username)){
				// If owner of the row info or an admin then display the edit task button

				$out .= '<td class="list_value" align="center"><a href="module.php?mod=research_assistant'.
					'&amp;action=edittask&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_edit"].'</a></td>'.
					'<td class="list_value" align="center"><a href="module.php?mod=research_assistant'.
					'&amp;action=deletetask&amp;taskid='.$row->t_id.'">'.$pgv_lang["delete"].'</a></td>'.
					'<td class="list_value" align="center"></td>';
				} else {
					$out .= '<td class="list_value" align="center">&nbsp;</td>'.
						'<td class="list_value" align="center">&nbsp;</td>'.
						'<td class="list_value" align="center">&nbsp;</td>';
				}
				$out .= '</tr>';
			}
		}
		$out .= '</table>';
		return $out;
	}

	/**
	 * Return a Root folder from the database (ie fr_parentid IS NULL)
	 * Note that to get a next level folder call get_subfolders($parentId)
	 *
	 * @param string $orderby If passed, query the database to order by
	 * @return mixed Results from the database
	 */
	function get_folder($orderby) {
		// Obtain the table prefix from the site
		global $TBLPREFIX;

		// Switch the orderby clause
		if (!empty ($orderby)) {
			$this->switch_order();
			$sql = "SELECT * FROM {$TBLPREFIX}folders WHERE fr_parentid IS NULL ORDER BY ".$orderby." ".$_REQUEST['type'];
		} else {
			$sql = "SELECT * FROM {$TBLPREFIX}folders WHERE fr_parentid IS NULL";
		}

		return
			PGV_DB::prepare($sql)
			->fetchAll();
	}

	/*
	 * Returns if a folder currently has tasks in it and dissallows users from deleting it
	 * @param takes in a given folder id
	 *
	 * @returns true or false
	 */
	function folder_hastasks($folderid){
		global $TBLPREFIX;

		return $folderid &&
			PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_fr_id=?")
			->execute(array($folderid))
			->fetchOne();
	}

	/*
	 * Returns if a folder currently has folders in it and dissallows users from deleting it
	 * @param takes in a given folder id
	 *
	 * @returns true or false
	 */
	function folder_hasfolders($folderid){
		global $TBLPREFIX;

		return $folderid &&
			PGV_DB::prepare("SELECT 1 FROM {$TBLPREFIX}folders WHERE fr_parentid=?")
			->execute(array($folderid))
			->fetchOne();
	}
	/**
	 * Switch the orderby clause - asc is ascending order, desc is descending order
	 *
	 * @return void
	 */
	function switch_order() {
		if ($_REQUEST["type"] == "asc") {
			$_REQUEST["type"] = "desc";
		} else {
			$_REQUEST["type"] = "asc";
		}
	}

	/**
	 * Returns all of the subfolders of a folder from the database
	 *
	 * @param int $parentId The id of the folder to look inn
	 * @param string $orderby - now ignored
	 * @return mixed Results from the database
	 */
	function get_subfolders($parentId, $orderby = "") {
		// Obtain the table prefix from the site
		global $TBLPREFIX;

		return
			PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}folders WHERE fr_parentid=?")
			->execute(array($parentId))
			->fetchAll();
	}

	/**
	 * Returns the the top folder from the database
	 *
	 * @param Int $folderId The folder to look for
	 * @return mixed Results from the database
	 */
	function get_top_folder($folderId) {
		global $TBLPREFIX;

		return
			PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}folders WHERE fr_id=?")
			->execute(array($folderId))
			->fetchOneRow();
	}

	/**
	 * Display the top folder image and name
	 *
	 * @param Int $folderId The folder you would like to print at the top of the screenn
	 * @return mixed HTML to print the top folder
	 */
	function print_top_folder($folderId) {
		$out = "";

		if (!empty ($folderId)) {
			$row=$this->get_top_folder($folderId);

			$out = '<img src="modules/research_assistant/images/folder_blue_icon.gif" alt="Folder"></img>';
			$out .= "<strong> ".PrintReady($row->fr_name)."</strong>";
		}
		return $out;
	}

	/**
	 * print_folder_view
	 *
	 * @param int $folderId If passed, Gets the subfolders
	 * @param string $orderby If passed, Query the database with an order by clause
	 * @return string HTML to print out the folder view
	 */
	function print_folder_view($folderId = "", $orderby = "") {
		global $pgv_lang, $TBLPREFIX;
		$out = "";

		if (!isset ($_REQUEST["type"])) {
			$_REQUEST["type"] = "asc";
		}

		// Find the folder or sub folders
		if (empty ($folderId)) {
			$rows = $this->get_folder($orderby);
		} else {
			$rows = $this->get_subfolders($folderId, $orderby);
		}

		// Print the Folder table header and colum header block
		if ($rows) {
			$out .= '<table class="list_table" align="center" border="0" width="700">';
			$out .= '<thead><tr><th colspan="3" class="topbottombar"><h2>'.
				print_help_link("ra_folder_view_help", "qm", '', false, true).$pgv_lang["ra_Folder_View"].'</h2></th></tr>';
			$out .= '<tr><th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderbyfolder=fr_name'.
				'&amp;type='.$_REQUEST["type"].'">'.$pgv_lang["ra_Folder_Name"].'</a></th>';
			$out .= '<th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderbyfolder='.
				'fr_description&amp;type='.$_REQUEST["type"].'">'.$pgv_lang["ra_description"].'</a></th>';
			$out .= '<th class="list_label">'.$pgv_lang["ra_edit"].'</th></tr><thead>';
		}

		// Print the table of folders
		foreach ($rows as $row) {
			$out .= '<tr><td class="optionbox"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$row->fr_id.'"><img src='.
				'"modules/research_assistant/images/folder_blue_icon.gif" border="0" alt='.
				'"Folder" style="vertical-align: middle;"></img> '.PrintReady($row->fr_name).'</a></td>'.
				'<td class="optionbox wrap">'.nl2br(PrintReady($row->fr_description)).'</td>'.
				'<td class="optionbox" align="center"><a href="module.php?mod=research_assistant'.
				'&amp;action=editfolder&amp;folderid='.$row->fr_id.'">'.$pgv_lang["ra_edit"].'</a></td>
				</tr>';
		}
		$out .= '</table>';
		return $out;
	}

	/**
	 * Deletes a folder
	 */
	function deleteFolder($folderid) {
		global $TBLPREFIX;

		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}folders WHERE fr_id=?")
			->execute(array($folderid));
	}

	/**
	 * Display the tasks to the users
	 *
	 * @param int $folderId
	 * @param string $res Result from the database
	 * @return string HTML to print out the folder view
	 */
	function print_tasks($folderId, $rows) {
		global $pgv_lang;
		$out = "";

		// Display tasks if we're inside of a folder - Display header and column bar first
		if (!empty ($folderId)) {
			$out .= '<table id="Tasks" class="list_table" align="center" width="700" border="0">';
			$out .= '<thead><tr><th colspan="7" class="topbottombar"><h2>'.
				print_help_link("ra_view_tasklist_help", "qm", '', false, true).$pgv_lang["ra_Task_View"].'</h2></th></tr>';
			$out .= '<tr><th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderby=t_title&amp;type='.
				$_REQUEST["type"].'">'.$pgv_lang["ra_title"].'</a></th>'.
				'<th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderby=t_description&amp;type='.
				$_REQUEST["type"].'">'.$pgv_lang["ra_description"].'</a></th>'."\n".
				'<th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderby=t_startdate&amp;type='.
				$_REQUEST["type"].'">'.$pgv_lang["ra_Start_Date"].'</a></th>'."\n".
				'<th class="list_label"><a href="module.php?mod=research_assistant'.
				'&amp;action=viewtasks&amp;folderid='.$folderId.'&amp;orderby=t_enddate&amp;type='.
				$_REQUEST["type"].'">'.$pgv_lang["ra_completed"].'</a></th>'."\n".
				'<th class="list_label">'.$pgv_lang["ra_assigned"].'</th>'.
				'<th class="list_label">'.$pgv_lang["ra_edit"].'</th><th class="list_label">'.
				$pgv_lang["ra_delete"].'</th></tr></thead>'."\n";
		}

		// Loop through the database results and print a row for each task
		foreach ($rows as $row) {
			$out .= '<tr><td class="list_value"><a href="module.php?mod=research_assistant&amp;action='.
				'viewtask&amp;taskid='.$row->t_id.'"class="list_item name2">'.PrintReady($row->t_title).'</a></td>';
			$date=timestamp_to_gedcom_date($row->t_startdate);
			$out .= '<td class="list_value_wrap">'.PrintReady($row->t_description).'</td>';
			$out .= '<td class="list_value" align="center">'.$date->Display(false).'</td>';
			$out .= '<td class="list_value" align="center">'.$this->checkComplete($row).'</td>';
			$out .= '<td class="list_value" align="center">';
//			if (!empty($row->t_username)) $out .= $pgv_lang["ra_yes"];
//			else $out .= $pgv_lang["ra_no"];
			$out .= $row->t_username;

			if(PGV_USER_IS_ADMIN || PGV_USER_NAME==($row->t_username)){
			// If owner of the row info or an admin then display the edit and delete task links

				$out .= '<td class="list_value" align="center"><a href="module.php?mod=research_assistant&amp;action='.
					'edittask&amp;taskid='.$row->t_id.'">'.$pgv_lang["ra_edit"].'</a></td>';
				$out .= '<td class="list_value" align="center"><a href="module.php?mod=research_assistant&amp;action='.
					'deletetask&amp;taskid='.$row->t_id.'&amp;folder='.$folderId.'">'.$pgv_lang["ra_delete"].'</a></td>';
			}
			else {
				$out .= '<td class="list_value" align="center">&nbsp;</td>'.
					'<td class="list_value" align="center">&nbsp;</td>';
			}
			$out .= '</tr>';
		}
		$out .= '</table>';

		return $out;
	}

	/*
	 * Deletes a task
	 * When a task is deleted then not only is the task deleted but, all comments related to the task, 
	 *  while all links to the associated sources, facts and individuals are also removed
	 */
	function deleteTask($taskid) {
		global $TBLPREFIX;

		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}tasks WHERE t_id=?")
			->execute(array($taskid));
		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}comments WHERE c_t_id=?")
			->execute(array($taskid));
		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}tasksource WHERE ts_t_id=?")
			->execute(array($taskid));
		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}taskfacts WHERE tf_t_id=?")
			->execute(array($taskid));
		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}individualtask WHERE it_t_id=?")
			->execute(array($taskid));
	}

	/**
	 * Scans the form dir and prints out all of the forms that we have to add information with
	 *
	 * @param int $pid The person id we need to be linking information to
	 * @param bool $links Tells the function to kick out links or just text
	 * @return mixed The list of links for the forms or false
	 */
	function print_form_list($links = false) {
		$out = '';

		// Display a list of forms for the user
		if ($handle = opendir(BASEPATH."/forms")) {
			while (($file = readdir($handle)) !== false) {
				if (strpos($file, ".")) {
					list ($file, $ext) = explode(".", $file);
					// Make sure we don't show the . and .. nor any forms
					// that start with ra_ as those are forms that we use internally
					if ($file != "." && $file != ".." && !is_dir($file) && 
						(strstr($file, "ra_") === false) && (strstr($file, "FormBuilder") === false)) {
						// Setup pretty print for the users
						$fileNoSpaces = preg_replace('/\s+/', '_', $file);
						$fileSpaces = str_replace('_', ' ', $file);

						// Print a link with all the information to connect a person and their data.
						if ($links)
							$out .= '<a href="module.php?mod=research_assistant&amp;action='.
								'printform&amp;formname='.$fileNoSpaces.'">'.$fileSpaces.'</a><br/>';
						else
							$out .= $fileSpaces."\n";
					}
				}
			}
			closedir($handle);
		}

		// list of forms is not garanteed in order, so sort it before returning it
		$temp = explode("\n", $out);
		sort($temp);
		$out = implode("\n", $temp);
		return $out;
	}

	/**
	 * Call any function inside of the form
	 *
	 * @param string $name Name of the class (Form) to create
	 * @param string $func Which function you want to call
	 * @param mixed $args Any arguments to the function
	 * @return mixed
	 */
	function form_function($name = '', $func = '', $args = null) {
		if (empty ($func) || empty ($name))
			return false;

		// Add the php extension if it's not there
		if (strstr($name, ".php") === false)
			$ext = ".php";

		$path = BASEPATH."/forms/".$name.$ext;
		$out = '';

		if (file_exists($path)) {
			// Perform the function
			require_once $path;
			$form = new $name ();
			$out = $form-> $func ($args);
		}
		return $out;
	}

	/**
	* Function that checks to see if a task is complete.
	*
	* @param task Supply this with a DB row of a task, it will auto check to see if it's completed.
	* @return mixed
	*/
	function checkComplete($task) {
		// Globals
		global $pgv_lang;

		// If there is no end date the task is not complete.
		if (empty ($task->t_enddate))
			return $pgv_lang["ra_no"];
		// If there is an end date, it is complete.
		else
			return $pgv_lang["ra_yes"];
	}

	/**
	* Function to complete a task
	*
	* @param taskid for the task to be completed
	*/
	function completeTask($taskid, $form='') {
		global $TBLPREFIX;

		PGV_DB::prepare("UPDATE {$TBLPREFIX}tasks SET t_enddate=?, t_form=? WHERE t_id=?")
			->execute(array(time(), $form, $taskid));
	}

	/**
	 * Function to find missing information
	 * Searches for following missing info SEX, BIRT, DEAT, GIVN (given name), SURN (surname)
	 * @param Person $person
	 * @return mixed array of GEDCOM tags and a related multi-dimensional array of the probabilities and fact types
	 *	This multi-dimensional array contains two more arrays, one for global inferences and one for local inferences
	 * 	Both the global and local contain first the probability, and second the value of what you are looking for
	 */
	function getMissinginfo(& $person) {
		global $factarray, $pgv_lang;

		$ra_nonplacfacts = array("ENDL","NCHI","SLGC","SLGS","SSN","CHAN","_UID","ADDR","PHON","EMAIL");
		$ra_templefacts = array("SLGC","SLGS","BAPL","ENDL","CONL");
		$ra_nondatefacts = array("ABBR","ADDR","AFN","AUTH","EMAIL","FAX","NAME",
					"NCHI","NOTE","OBJE","PHON","PUBL","REFN","REPO",
					"SEX","SOUR","SSN","TEXT","TITL","WWW","_EMAIL","_UID");
		$ra_emptyfacts = array("BIRT","CHR","DEAT","BURI","CREM","ADOP","BAPM","BARM",
					"BASM","BLES","CHRA","CONF","FCOM","ORDN","NATU","EMIG",
					"IMMI","CENS","PROB","WILL","GRAD","RETI","BAPL","CONL",
					"ENDL","SLGC","EVEN","MARR","SLGS","MARL","ANUL","CENS",
					"DIV","DIVF","ENGA","MARB","MARC","MARS","CHAN","_SEPR",
					"RESI", "DATA", "MAP");

		$perId = $person->getXref();
		$DontCheckOfSubAll = "";
		$MissingReturn = array ();  //Local var for the return string

		// check for missing sex info
		if ($person->getSex() == "U") 
		{
			$MissingReturn[] = array("SEX", $pgv_lang["ra_All"]);
		}

		// check for missing birth info
		$birtdate=$person->getBirthDate();
		$birtplac=$person->getBirthPlace();
		if (!$birtplac && !$birtdate->isOK()) { 
			$probFacts = singleInference($perId,"BIRT");
			$MissingReturn[] = array("BIRT", $pgv_lang["ra_All"],$probFacts);
			$DontCheckOfSubAll .= "BIRT";
		}

		// check for missing death info
		$deatdate=$person->getDeathDate();
		$deatplac=$person->getDeathPlace();
		if ((!$deatplac && !$deatdate->isOK()) && $person->isDead()) { 
			$probFacts = singleInference($perId,"DEAT");
			$MissingReturn[] = array("DEAT", $pgv_lang["ra_All"], $probFacts);
			$DontCheckOfSubAll .= "DEAT";
		}

		// check for given and surname info
		list($lastname, $givennames)=explode(',', $person->getSortName());
		if (substr($givennames,0,1)=='@') {
			$probFacts = singleInference($perId,"GIVN");
			$MissingReturn[] = array("GIVN","",$probFacts);
		}
		if (substr($lastname,0,1)=='@') {
			$probFacts = singleInference($perId,"SURN");
			$MissingReturn[] = array("SURN","",$probFacts);
		}

		// check for missing burial info but only if person is already dead
		$cemefound = "";
		if ($person->isDead()) {
			$buriEvent = $person->getFactByType('BURI');
			if (empty($buriEvent)) { 	
				$probFacts = singleInference($perId,"BURI");
				$MissingReturn[] = array("BURI", $pgv_lang["ra_All"],$probFacts);
				$DontCheckOfSubAll .= "BURI";
				// In case all burial info is missing ignore cemetery missing search
				$cemefound = $pgv_lang["ra_All"];
			}
		}

		// We now get all the fact information for an individual
		// and walk through these to look at inferences possible with them
		// For example if dates (DATE) are empty or EQ 0 they are considered missing,
		// 	if sources (SOUR) are empty they are considered missing
		//	if places (PLAC) are empty they are considered missing
		$indifacts = $person->getIndiFacts();
		/* @var $far Event */
		foreach ($indifacts as $key => $far) {
			$fact = $far->getTag();

			// if this key and data are part of all missing flag then don't check any further
			if (str_contains($DontCheckOfSubAll, $fact)) continue;

			$event = $far->getDetail();
			if ($fact=="EVEN" || $fact=="FACT") {
				$fact = $far->getType();
			}
			// Check if whole date is missing or any part (yr, mth, day)
			// If part of the date is missing then flag that in particular
			// Where date is missing in some form return the fact with date and year to make task titles unique
			$date = $far->getDate();
			if (empty ($date) || ($date->gregorianYear() == 0) || ($date->date1->m == 0) || ($date->date1->d == 0)) {
				if (!in_array($fact, $ra_nondatefacts)) {
					$whatsmissing = "DATE";
					if (!empty($date) && !empty ($date->date1)) {
						$strdate = "";
						if ($date->date1->d == 0)  $strdate .= "dd "; else $strdate .= sprintf("%4d",$date->date1->d)." ";
						if ($date->date1->m == 0)  $strdate .= "mmm "; else $strdate .= sprintf("%2d",$date->date1->m)." ";
						if ($date->gregorianYear() == 0) $strdate .= "yyyy"; else $strdate .= sprintf("%4d",$date->gregorianYear());
						if (strlen($strdate > 0)) $whatsmissing .= "</br>".str_replace("%INF_FACT%",$strdate,$pgv_lang["ra_IncompleteDate"]);
						else if ($date->gregorianYear() != 0) $whatsmissing .=  " ".$date->gregorianYear();
					}
					$MissingReturn[] = array ($fact, $whatsmissing);
				}
			}
			$source = get_gedcom_value("SOUR", 2, $far->getGedComRecord());
			if (empty ($source) && !in_array($fact, $ra_nonplacfacts) && !in_array($fact, $ra_nondatefacts))
				$MissingReturn[] = array ($fact, "SOUR");
			$plac = $far->getPlace();
			if (empty ($plac)) {
				if (in_array($fact, $ra_templefacts)) {
					$plac = $far->getValue("TEMP");
					if (empty($plac)) {
						$MissingReturn[] = array ($fact, "TEMP");
					}
				}
				else {
					if (!in_array($fact, $ra_nonplacfacts)) {
						$probFacts = singleInference($perId,"$fact:PLAC");
						$MissingReturn[] = array ($fact, "PLAC",$probFacts);
					}
				}
			}
			$cemeinfo = get_gedcom_value("CEME", 2, $far->getGedComRecord());
			if (!empty($cemeinfo)) $cemefound = $cemeinfo;
		}

		// Special case after all other facts check for missing burial cemetery info but only if person is already dead
		// 
		if ($person->isDead() && ($cemefound == "")) {
			$probFacts = singleInference($perId,"BURI:CEME");
			$MissingReturn[] = array("BURI", $pgv_lang["ra_autosearch_cemetery"],$probFacts);
		}
		return $MissingReturn; //return of missing info check results empty string if no missing info
	}
	//End of the missing info functions

	/**
	 * Creates a new individual in the database
	 */
	function add_indi_task($tid, $itid, $ids) {
		global $TBLPREFIX;

		PGV_DB::prepare("INSERT INTO {$TBLPREFIX}individualtask (it_t_id, it_i_id, it_i_file) VALUES (?, ?, ?)")
			->execute(array($tid, $itid, $ids));
	}

	/**
	 * Adds a souce to an fact
	 */
	function add_sources($task_id, $sources = -1) {
		global $TBLPREFIX;

		if (!is_array($sources)) {
			$sources = explode(';', $sources);
		}

		foreach ($sources as $source_id) {
			if (!empty($source_id)) {
				PGV_DB::prepare("INSERT INTO {$TBLPREFIX}tasksource (ts_t_id, ts_s_id) VALUES (?, ?)")
					->execute(array($task_id, $source_id));
				//-- only allow one source
				break;
			}
		}
	}

	/**
	 * Add a task into the database with all of the required information
	 * If a username is not passed then assign the task to the currently logged in user
	 * If successfully add task then return true, otherwise return false
	 */
	function add_task($taskid, $folder, $title, $description, $pid, $userName=PGV_USER_NAME) {
		global $TBLPREFIX;

		$exists=
			PGV_DB::prepare("SELECT 1 FROM {$TBLPREFIX}tasks, 
				{$TBLPREFIX}individualtask WHERE t_title=? AND it_t_id=t_id AND it_i_id=? AND it_i_file=?")
			->execute(array($title, $pid, PGV_GED_ID))
			->fetchOne();

		//make sure the same task does not exist already so we can add an individual task
		if (!$exists) {
			PGV_DB::prepare("INSERT INTO {$TBLPREFIX}tasks (t_id, t_fr_id, t_title, t_description, t_startdate, t_username) VALUES (?, ?, ?, ?, ?, ?)")
				->execute(array($taskid, $folder, $title, $description, time(), $userName));
			return true;
		} else {
			return false;
		}

	}

	/**
	 * Make sure that they are not trying to add no tasks, then loop through and 
	 * all selected tasks and add them into the task and individual task tables
	 * Use the forms missingName array of checked items to pass what Task title to use
	 */
	function auto_add_task(&$person,$folderID) {
		global $TBLPREFIX;

		if(!empty($_REQUEST['missingName'])){
			$missingName = $_REQUEST['missingName'];
			for($i=0; $i<count($missingName); $i++) {
				$nextTaskID =get_next_id("tasks","t_id");
				$personName = $person->getFullName();

				// Add a task and if sucessful (true) then at the indi person relation to that task
				if ($this->add_task($nextTaskID,$folderID,$missingName[$i],$personName." (auto generated)",$person->getXref())) {
					$this->add_indi_task($nextTaskID,$person->getXref(), PGV_GED_ID);
				}
			}

			// After auto adding all these tasks go back to the individuals RA tab - current method doesn't work!
//			print "<script language=\"JavaScript\">window.location = individual.php?pid=".$person->getXref()."&tab=6</script>";
		}
	}

	/**
	 * trims a PLAC string to a certain depth for comparison purposes
	 * used withing the inferences function
	 */
	function trimLocation($loc) {
		$loclevels = 4;  // number of levels of detail to keep
		if (!$loc) return "";
		$newLoc = "";
		// reverse the array so that we get the top level first
		$levels = array_reverse(explode(',', $loc));
		foreach($levels as $pindex=>$ppart) {
			// build the location string in reverse order, up to the requested number of levels
			if ($pindex < $loclevels) $newLoc .= trim($ppart).",";
		}
		// there is an extra comma at the end, but since this string is just used for comparison purposes, it won't hurt anything
		return ($newLoc);
	}

	/**
	 * The inferences function will look for correlations and return an array
	 * with each probability so, it will have a meaningful index system such
	 * as ['FATHER:BIRT:PLAC'] to return the probability that an individual will
	 * have the same birth place as their father or ['BIRT:MARR:DEAT'] for the
	 * probability that an individual will have the same birth, marriage and
	 * death place at each index of the array will be a description as well as a
	 * percentage liklihood of the given correlation.
	 */
	function inferences() {
		global $TBLPREFIX;

		require_once PGV_ROOT.'modules/research_assistant/ra_ViewInferencesArray.php';
		$indilist = get_indilist_indis();

		foreach ($indilist as $indi) {
			//assign surname, gender, birthplace and occupation for the individual
			$gender = $indi->getSex();
			$locals = array();
			foreach($inferences as $pr_id=>$value) {
				//-- get the local value from the the individual record
				if (!isset($locals[$value['local']])) {
					switch ($value['local']) {
					case 'GIVN':
					case 'SURN':
						list($locals['SURN'], $locals['GIVN'])=explode(',', $indi->getSortName());
						break;
					default:
						$gedvalue = get_gedcom_value($value['local'], 1, $indi->getGedcomRecord(), '', false);
						if ($gedvalue && strpos($value['comp'],':PLAC') !== false) {
							// this is a PLAC string, trim it to a consistent number of levels
							$gedvalue = ra_functions::trimLocation($gedvalue);
						}
						$locals[$value['local']] = $gedvalue;
						break;
					}
				}

				//-- load up the gedcom record we want to compare the data from
				//-- record defaults to the indis record, after this section runs it will be
				//-- set to the record from the inferences table that we want to compare the value to
				$record = $indi;
				if ($record) {
					$rec_tags = explode(':', $value['record']);
					while ($record && $rec_tags) {
						$tag = array_shift($rec_tags);
						if (preg_match("/1 $tag @(.*)@/", $record->getGedcomRecord(), $match)) {
							if ($tag=='FAMS' && $rec_tags && $rec_tags[0]=='SPOUSE') {
								$record=$record->getCurrentSpouse();
								array_shift($rec_tags);
							} else {
								$record=GedcomRecord::getInstance($match[1]);
							}
						} else {
							$record=null;
						}
					}
				}

				if ($record) {
					switch ($value['comp']) {
					case 'SURN':
						list($surn)=explode(',', $record->getSortName());
						if ($surn==$locals['SURN'] && $surn!='@N.N.') {
							$inferences[$pr_id]['value']++;
						}
						break;
					case 'GIVN':
						list($dummy, $givn)=explode(',', $record->getSortName());
						$parts1 = preg_split("/\s+/", $givn);
						$parts2 = preg_split("/\s+/", $locals['GIVN']);
						foreach ($parts1 as $part1) {
							foreach ($parts2 as $part2) {
								if ($part1==$part2 && $part1!='@P.N.') {
									$inferences[$pr_id]['value']++;
									break 2;
								}
							}
						}
						break;
					default:
						$gedvalue = get_gedcom_value($value['local'], 1, $indi->getGedcomRecord(), '', false);
						if ($gedvalue && strpos($value['comp'],':PLAC') !== false) {
							// this is a PLAC string, trim it to a consistent number of levels
							$gedvalue = ra_functions::trimLocation($gedvalue);
						}
						if ($gedvalue && UTF8_strtolower($gedvalue)==UTF8_strtolower($locals[$value['local']])) {
							$inferences[$pr_id]['value']++;
						}
						break;
					}
					$inferences[$pr_id]['count']++;
				}
			}
		}

		/*
		 *The following section is used to store the calculated percentages in the database
		 */
		PGV_DB::prepare("DELETE FROM {$TBLPREFIX}probabilities WHERE pr_file=?")
			->execute(array(PGV_GED_ID));

		/**
		 * pr_id int unsigned NOT NULL auto_increment,
		 * pr_f_lvl varchar(200) NOT NULL,
		 * pr_s_lvl varchar(200),
		 * pr_rel varchar(200) NOT NULL,
		 * pr_matches int
		 * pr_count int
		 * pr_file INT
		 */
		foreach ($inferences as $pr_id=>$value) {
			PGV_DB::prepare("INSERT INTO 
				{$TBLPREFIX}probabilities (pr_id, pr_f_lvl, pr_s_lvl, pr_rel, pr_matches, pr_count, pr_file) 
				VALUES (?, ?, ?, ?, ?, ?, ?)")
			->execute(array(get_next_id("probabilities", "pr_id"), $value['local'], $value['record'],
				 $value['comp'], $value['value'], $value['count'], PGV_GED_ID));
		}
		return $inferences;
	}

	/*
	 * Check to see which of the suggested research tasks have already been added into the database
	 */
	function task_check($title, $pid) {
		global $TBLPREFIX;

		return
			PGV_DB::prepare("SELECT t_id FROM {$TBLPREFIX}tasks, 
				{$TBLPREFIX}individualtask WHERE t_title=? AND it_t_id=t_id AND it_i_id=? AND it_i_file=?")
			->execute(array($title, $pid, PGV_GED_ID))
			->fetchOne();
	}

	/*
	 * Call the already created get_folder() function and for each one found create a new option tag to insert into the
	 * select tag, with only the first item marked as selected.
	 */
	function folder_search() {
		global $TBLPREFIX, $pgv_lang;

		$selectedset = 0;
		//TODO: Figure out how to display folder Heirarchies correctly
		$rows = $this->get_folder("");
		$out = "";
		foreach ($rows as $row) {
			$out .= "<option value=".$row->fr_id;
			if ($selectedset == 0) {
				$out .= " selected=selected";
				$selectedset =1;
			}
			$out .=">".$row->fr_name."</option>";
		}
		return $out;
	}

	/**
	 * Get Tasks for Source
	 */
	function getSourceTasks($sId, $legend="") {
		global $pgv_lang, $TBLPREFIX, $TEXT_DIRECTION;

		$legend = '<img src="modules/research_assistant/images/view_inferences.gif" alt="" align="middle" /> '.
			$pgv_lang["ra_task_list"].' @ '.$legend;
		$out = '<fieldset><legend>'.$legend.'</legend>';
		$out .= '<div id=ra-table" class="'.$TEXT_DIRECTION.'">';
		$rows=
			PGV_DB::prepare("SELECT t_id, t_title, t_description, t_startdate, t_enddate, t_username FROM 
				{$TBLPREFIX}tasks, {$TBLPREFIX}tasksource, {$TBLPREFIX}sources 
				WHERE t_id=ts_t_id AND s_id=ts_s_id AND s_id=? AND s_file=?")
			->execute(array($sId, PGV_GED_ID))
			->fetchAll();

		if (!$rows) {
			$out .= '<table width="100%"><tr>';
			$out .= '<thead><th class="topbottombar">'.print_help_link("ra_sources_task_list_help", "qm", '', false, true).
				$pgv_lang["ra_task_list"].'</th></tr>'."\n";
			$out .= '<tr><td class="list_value" colspan="4" align="center">'.$pgv_lang["ra_no_sour_tasks"].'</td></tr></thead>'."\n";
		} else {
			$n = 0;
			$out .= '<table class="list_table">';
			$out .= '<thead><tr><td></td><th class="topbottombar" colspan="4">'.
				print_help_link("ra_sources_task_list_help", "qm", '', false, true).$pgv_lang["ra_task_list"].'</th></tr>'."\n";
			// Loop through all the task ID's and pull the info we need on them,
			// then format them nicely to show the user.
			foreach ($rows as $row) {
				if ($row && empty($row->t_enddate)) {
					if ($n == 0) {
						$out .= '<tr><td></td><th class="list_label">'.$pgv_lang["ra_title"].'</th>';
						$out .= '<th class="list_label">'.$pgv_lang["ra_description"].'</th>';
						$out .= '<th class="list_label">'.$pgv_lang["ra_assigned"].'</th>';
						$out .= '<th class="list_label">'.$pgv_lang["ra_created"].'</th></tr></thead>'."\n";
						$out .= '<tbody>';
					}

					$date=timestamp_to_gedcom_date($row->t_startdate);
					$out .= '<tr><td class="list_value_wrap rela list_item">'.++$n.'</td>';
					$out .= '<td class="list_value_wrap"><a href="module.php?mod=research_assistant'.
						'&amp;action=viewtask&amp;taskid='.$row->t_id.'" class="list_item name2">'.
						PrintReady($row->t_title).'</a></td>';
					if (UTF8_strlen($row->t_description)<=40) {
						$out .= '<td class="list_value_wrap">'.PrintReady($row->t_description).'</td>';
					}
					else {
						$out .= '<td class="list_value_wrap">'.
							UTF8_substr(PrintReady($row->t_description), 0, 40).$pgv_lang["ra_ellipsis"].'</td>';
					}
					$out .= '<td class="list_value_wrap center">'.PrintReady($row->t_username).'</td>';
					$out .= '<td class="list_value_wrap center">'.$date->Display(false).'</td></tr>';
				}
			}
			if ($n == 0) 
				$out .= '<tbody><tr><td></td><td class="list_value" colspan="4" align="center">'.$pgv_lang["ra_no_sour_tasks"].'</td></tr>'."\n";
			$out .= '</tbody>';
		}
		$out .= '</table></div></fieldset>';
		return $out;
	}

	/**
	 * Load up off site search names here
	 * Auto Search Plugin: To load up a new plugin follow the format for the two entries shown below
	 * ex $sites["myplugin.php"] = "mywebsite.com";
	 */
	function autoSearchFormOptions() {
		$this->sites = array();
		$this->sites["ancestry.php"] = "Ancestry.com";
		$this->sites["ancestrycouk.php"] = "Ancestry.co.uk";
		$this->sites["familysearch.php"] = "FamilySearch.org";
		$this->sites["geneanet.php"] = "GeneaNet.org";
		$this->sites["werelate.php"] = "Werelate.org";
		$this->sites["findagrave.php"] = "Findagrave.com";
		$this->sites["billiongraves.php"] = "Billiongraves.com";
// Comment out sites that are currently not working
//		$this->sites["genealogy.php"] = "Genealogy.com";
//		$this->sites["ellisisland.php"] = "EllisIslandRecords.org";
//		$this->sites["gensearchhelp.php"] = "Genealogy-Search-Help.com";
		$opts = "";
		$optCount = 1;
		//load up the options into the html
		foreach($this->sites as $key=>$value) {
			$opts .= "<option value=\"".$key."\" class=\"".$optCount."\">".$value."</option>\n";
			$optCount+=1;
		}
		return $opts;
	}

	function determineClosest(&$currentDate, $dateToCompare, $dateCompareAgainst )
	{
		if (!is_numeric($dateToCompare)) $dateToCompare = 0;
		if (!is_numeric($dateCompareAgainst)) $dateCompareAgainst = 0;

		if ($dateCompareAgainst > $dateToCompare)
			$compareDiff = $dateCompareAgainst - $dateToCompare;
		else
			$compareDiff = $dateToCompare - $dateCompareAgainst;

		if ($dateCompareAgainst > $currentDate)
			$currentDiff = $dateCompareAgainst - $currentDate;
		else
			$currentDiff = $currentDate - $dateCompareAgainst;

		if ($compareDiff < $currentDiff)
			return $dateToCompare;
		else
			return $currentDate;
	}

	/**
	 * tab is the function that builds the display for the different screens.
	 * These screens are identified by a tab
	 * @param Person $person
	 */
	function tab(&$person) {
		// Start our engines.
		global $pgv_lang, $TBLPREFIX;
		global $factarray;

		if (!is_object($person)) return "";

		$bdate = $person->getEstimatedBirthDate();
		if (is_null($bdate)) $byear = '';
		else $byear = $bdate->gregorianYear();
 
		$ddate = $person->getEstimatedDeathDate();
		if (is_null($ddate)) $dyear = '';
		else $dyear = $ddate->gregorianYear();

		// add an individuals task if it was queued up by calling auto_add_task
		if (isset ($_REQUEST['action']) && $_REQUEST['action'] == 'ra_addtask') {
			$this->auto_add_task($person, $_POST['folder']);
		}
		// gets task id from the database so we can populate the tasks table for this individual
		try {
			$rows=
				PGV_DB::prepare("SELECT it_t_id FROM {$TBLPREFIX}individualtask WHERE it_i_id=? AND it_i_file=?")
				->execute(array($person->getXref(), PGV_GED_ID))
				->fetchAll();
		} catch (PDOException $ex) {
			return '<span class="error">There was an error with the Research Assistant database.  '.
				'Click on the <a href="module.php?mod=research_assistant">Research Assistant'.
				'</a> icon to create the database.<br />&nbsp;</span>';
		}

		// Start of HTML output for individual task list
		$out = "\n\t".'<table class="list_table">';
		$out .= '<thead><tr><td></td><td class="topbottombar" colspan="4" align="center">'.print_help_link("ra_person_task_list_help", "qm", '', false, true).
			'<b>'.$pgv_lang["ra_task_list"].'</b></td></tr>';

		$n = 0;
		// Loop through all the task ID's and pull the info we need on them,
		// then format them nicely to show the user.
		foreach ($rows as $row) {
			$task=
				PGV_DB::prepare("SELECT * FROM {$TBLPREFIX}tasks WHERE t_id=?")
				->execute(array($row->it_t_id))
				->fetchOneRow();

			if ($task && empty($task->t_enddate)) {
				if ($n == 0) {
					// print out the column titles
					$out .= "\n\t\t".'<tr><td></td><th class="list_label">'.$pgv_lang["ra_title"].'</th>';
					$out .= '<th class="list_label">'.$pgv_lang["ra_description"].'</th>';
					$out .= '<th class="list_label">'.$pgv_lang["ra_assigned"].'</th>';
					$out .= '<th class="list_label">'.$pgv_lang["ra_created"].'</th></tr></thead>'."\n";
					$out .= '<tbody>';
				}

				$date=timestamp_to_gedcom_date($task->t_startdate);
				$out .= '<tr><td class="list_value_wrap rela list_item">'.++$n.'</td>';
				$out .= '<td class="list_value_wrap">';
				$out .= '<a href="module.php?mod=research_assistant'.
					'&amp;action=viewtask&amp;taskid='.$task->t_id.'" class="list_item name2">'.
					PrintReady($task->t_title).'</a></td>';
				if (UTF8_strlen($task->t_description)<=40) {
					$out .= '<td class="list_value_wrap">'.PrintReady($task->t_description).'</td>';
				}
				else {
					$out .= '<td class="list_value_wrap">'.
						UTF8_substr(PrintReady($task->t_description), 0, 40).$pgv_lang["ra_ellipsis"].'</td>';
				}

				$out .= '<td class="list_value_wrap center">'.PrintReady($task->t_username).'</td>';
				$out .= '<td class="list_value center">'.$date->Display(false).'</td></tr>';
			}
		}
		if ($n == 0) 
			$out .= '<tr><td></td><td class="list_value" colspan="4" align="center">'.$pgv_lang["ra_no_indi_tasks"].'</td></tr></thead>';
		$out .= '</tbody>';

		// Create a new task button at bottom of the task list
		$addptaskfrm2Action = "module.php?mod=research_assistant&amp;action=addtask&amp;pid=".$person->getXref();
		$out .= '<tr><td></td><td class="topbottombar" colspan="4">
			<form action="'.$addptaskfrm2Action.'" method="post" name="addptaskfrm2">
			    <input type="hidden" name="mod" value="research_assistant" />
			    <input type="hidden" name="action" value="addtask" />
			    <input type="hidden" name="pid" value="'.$person->getXref().'" />
			    <input type="submit" value="'. $pgv_lang["ra_task_entry"] . '" />
			</form>
			</td></tr></table>'."\n";

// Older code used a URL to create a new task button - leave for moment but to be ripped later
//		$out .= '<tr><td></td><td class="topbottombar" colspan="4"><a href="module.php?mod=research_assistant'.
//			'&amp;action=addtask&amp;pid='.$person->getXref().'">'.$pgv_lang["ra_task_entry"].
//			'</a></td></tr></table>'."\n";

		// Beginning of the missing information table, which gets populated with missing information for 
		// that individual and allows the user to "autoadd" tasks.  A checkbox to view link conversion is
		// included if a piece of missing information is already auto tasked
		$Missing = $this->getMissinginfo($person);
		$out .='<table align="center"><tr><td valign="top">';
		$out .='<form name="autotask" action="individual.php?pid='.$person->getXref().'&amp;action=ra_addtask" method="post">
			<table border="0">
			<input type="hidden" name="pid" value="'.$person->getXref().'" />
			<input type="hidden" name="action" value="ra_addtask" />
			<tr>
				<td align="center" colspan="2" class="topbottombar">'.
					print_help_link("ra_missing_info_help", "qm", '', false, true).'<b>'.$pgv_lang['ra_missing_info'].
				'</td>
			</tr>';

		// All missing information is based on birth and death year, so retreive estimates of those first
		$bdatea = $person->getEstimatedBirthDate();
		if (is_null($bdatea)) $bdate = '';
		else {
			$bdatea = $bdatea->MinDate();
			if (is_null($bdatea)) $bdate = '';
			else {
				$bdatea = $bdatea->convert_to_cal('gregorian');
				$bdate  = $bdatea->Format('Y');
				$bdate .= ($bdatea->m) ? $bdatea->Format('m') : '00';
				$bdate .= ($bdatea->d) ? $bdatea->Format('d') : '00';
			}
		}

		$ddatea = $person->getEstimatedDeathDate();
		if (is_null($ddatea)) $ddate = '';
		else {
			$ddatea = $ddatea->MinDate();
			if (is_null($ddatea)) $ddate = '';
			else {
				$ddatea = $ddatea->convert_to_cal('gregorian');
				$ddate  = $ddatea->Format('Y');
				$ddate .= ($ddatea->m) ? $ddatea->Format('m') : '00';
				$ddate .= ($ddatea->d) ? $ddatea->Format('d') : '00';
			}
		}

		$sourcesInferred = array();
		$sourcesPrinted = array();

		// Every missing item gets a checkbox , so you can check it and make a task out of it
		// First we check the information that is already in an individuals record to show missing
		// information that should be there on existing GEDCOM tags
		// This may also generate some inferred sources of info which are put in $sourcesInferred array
		foreach ($Missing as $key => $val) 
		{
			$additionalInfer = array();
			$highest = 0;
			$factsExist = false;
			$compiled = "";
			$tasktitle = "";

			// Build the task title from the factarray info to 1) check if it exists so only display the task view
			// or 2) use the title to create an option for a new task
			if (isset($factarray[$val[0]])) {
				$tasktitle .= $factarray[$val[0]]." ";
			} elseif (isset($pgv_lang[$val[0]])) {
				$tasktitle .= $pgv_lang[$val[0]]." ";
			} else {
				$tasktitle .= $val[0]." ";
			}

			if (isset($factarray[$val[1]])) {
				$tasktitle .= $factarray[$val[1]];
			} elseif (isset($pgv_lang[$val[1]])) {
				$tasktitle .= $pgv_lang[$val[1]];
			} else {
				$tasktitle .= $val[1];
			}

			$taskid = $this->task_check($tasktitle, $person->getXref());
			if (!$taskid) // if the task_check passes, create a check box
			{
				$out .= '<tr><td width="20" class="optionbox"><input type="checkbox" name="missingName[]" value="'.
					$tasktitle.'" /></td><td class="optionbox">'."\n";
				$out .= "<span class=\"fact_label\">".$tasktitle."</span><br />";
				if(isset($val[2])) {
					foreach($val[2] as $inferKey=>$inferenceObj)
					{
						if($val[1] === $pgv_lang['ra_All'] || empty($val[1]) || $val[1] === "PLAC")
						{
							if(strstr($inferenceObj->getFactTag(),$val[0]) && $inferenceObj->getAverage() > 0)
							{
								if($inferenceObj->getFactValue() != "")
								{
									$additionalInfer[] = $inferenceObj;
								}
								if($highest < $inferenceObj->getAverage() && $inferenceObj->getFactValue() != "")
								{
									$compiled = array();
									$highest = $inferenceObj->getAverage();
									$compiled[0] = $this->decideInferSentence($inferenceObj->getAverage(),$inferenceObj->getFactTag(),$inferenceObj->getFactValue());

									if($inferenceObj->getFactTag() === "DEAT")
									{
										$posSources = $this->getEventsForDates($ddate - 5,$ddate+5,"",$inferenceObj->getFactValue());
									}
									else
									{
										if($inferenceObj->getFactTag() === "BIRT")
										{
											$posSources = $this->getEventsForDates($bdate-5,$bdate+5,"",$inferenceObj->getFactValue());
										}
										else
										{
											$posSources = $this->getEventsForDates($bdate,$ddate,"",$inferenceObj->getFactValue());
										}
									}
									if(count($posSources) > 0)
									{
										$compiled[0] .= '<br />';
										$compiled[0] .= $pgv_lang["ra_ThereIsChance"].": ";
										foreach($posSources as $sKey=>$sVal)
										{
											if(!in_array($sVal["id"],$sourcesInferred))
											{
												$sourcesInferred[$sVal["id"]] = $sVal;
											}
											$compiled[0] .= $sVal["description"]."<br />";
										}
									}

									$compiled[1] = $inferenceObj->getFactTag();
									$compiled[2] = $inferenceObj->getAverage();
									$compiled[3] = $inferenceObj->getFactValue();
								}
							}
						}
					}
					if(isset($compiled[0]))
					{
						$out .= $compiled[0];
					}
					if(!empty($additionalInfer))
					{
						$additionalFacts = false;
						$tempAdditional = "";
						foreach($additionalInfer as $addKey=>$addVal)
						{
							if($addVal->getFactValue() !== $compiled[3])
							{
								if($addVal->getFactTag() === "DEAT")
								{
									$posSources = $this->getEventsForDates($ddate - 5,$ddate+5,"",$addVal->getFactValue());
								}
								else
								{
									if($inferenceObj->getFactTag() === "BIRT")
									{
										$posSources = $this->getEventsForDates($bdate-5,$bdate+5,"",$addVal->getFactValue());
									}
									else
									{
										$posSources = $this->getEventsForDates($bdate,$ddate,"",$addVal->getFactValue());
									}
								}
								if(count($posSources) > 0)
								{
									$compiledSources = "";
									$compiledSources .= $pgv_lang["ra_ThereIsChance"].": ";
									foreach($posSources as $sKey=>$sVal)
									{
										if(!in_array($sVal["id"],$sourcesInferred))
										{
											$sourcesInferred[$sVal["id"]] = $sVal;
										}
										$compiledSources .= $sVal["description"]."<br />";
									}
								}
								$additionalFacts = true;
								$tempAdditional .= $this->decideInferSentence($addVal->getAverage(),$addVal->getFactTag(),$addVal->getFactValue());
								$tempAdditional .= '<br />';
								if(!empty($compiledSources)) $tempAdditional .= $compiledSources;
							}
						}
						if($additionalFacts)
						{
							$out .= '<br /><a href="" class="showit">'.$pgv_lang["ra_More"].'<span>';
							$out .= $tempAdditional;
						}
					}
				}
				$out .= "</td></tr>";

			} else // if not allow user to view the already created task
				$out .= '<tr><td width="20" class="optionbox"><a href="module.php?mod=research_assistant'.
					'&amp;action=viewtask&amp;taskid='.$taskid.'">'.$pgv_lang["ra_view"].'</a></td><td class="optionbox">'.
					$tasktitle.'</td></tr>'."\n";
		}

		// Next we are going to loop through the sources we have inferred from existing facts (now stored in $sourcesInferred)
		// For each one of these we generate an array for printing (to be stored in $sourcesPrinted)
		$factLookups = $this->getPlacesFromPerson($person);
		$person->add_family_facts(false);
		$tempDates = $person->getIndiFacts();

		foreach($sourcesInferred as $sKey=>$sVal)
		{
			$tasktitle = htmlentities($sVal["description"]);
			$taskid = $this->task_check($tasktitle, $person->getXref());
			if (!$taskid) // if the task_check passes, create a check box
			{
				$sourcesPrinted[$sVal["id"]] = $sVal;
				$out .= '<tr ><td width="20" class="optionbox">';
				$out .= '<input type="checkbox" name="missingName[]" value="'.$tasktitle.'" />';
				$out .= '<td class="optionbox">'.$sVal["description"];
				$out .= '</td></tr>';

			} else // if not allow user to view the already created task
				$out .= '<tr><td width="20" class="optionbox"><a href="module.php?mod=research_assistant'.
					'&amp;action=viewtask&amp;taskid='.$taskid.'">'.$pgv_lang["ra_view"].'</a></td><td class="optionbox">'.
					$tasktitle.'</td></tr>'."\n";

		}

		// Special clearance of all CENS records that might exist for this person should not be flagged as tasks
		foreach($tempDates as $tKey=>$tVal)
		{
			$tempTag = $tVal->getTag();
			if ($tempTag != "CENS") continue;
			$tempDate = $tVal->getDate()->gregorianYear();
			$tempPlace = (explode(",",$tVal->getPlace()));
			$tempCountry ="";
			foreach($tempPlace as $pKey=>$pVal) $tempCountry = trim($pVal);

			// Now we have this a valid CENS entry, compare to Facts list to exclude these going forward
			$events = $this->getEventsForDates($bdate,$ddate,"",$tempCountry);
			if(count($events) > 0)
			{
				foreach($events as $eventKey=>$eventVal)
				{
					if(!isset($sourcesPrinted[$eventVal["id"]]))
					{
						if (($eventVal["gedcom_fact"] == "CENS") && ($eventVal["pl_lv1"] == $tempCountry) && (substr($eventVal["startdate"],0,4) == $tempDate)) {
							$sourcesPrinted[$eventVal["id"]] = $eventVal;
						}
					}
				}
			}
		}

		// Special clearance of all _MILI records that might exist for this person should not be flagged as tasks
		// Note that military facts should assume that the individual was old enough to
		// participate, so in the case of events with gedcom_fact="_MILI" the birth date should be
		// minimum of real birth date + 15 years
		foreach($tempDates as $tKey=>$tVal)
		{
			$tempTag = $tVal->getTag();
			if ($tempTag != "_MILI") continue;
			$tempDate = $tVal->getDate();
			$tempDate = $tempDate->date1;
			if (is_object($tempDate)) {
				$cmpDate = $tempDate->y;
				$cmpDate.=($tempDate->m) ? $tempDate->Format('m') : '00';
				$cmpDate.=($tempDate->d) ? $tempDate->Format('d') : '00';
			} else $cmpDate = '';
			
			$tempPlace = (explode(",",$tVal->getPlace()));
			$tempCountry ="";
			foreach($tempPlace as $pKey=>$pVal) $tempCountry = trim($pVal);

			// Now we have this a valid MILI entry, compare to Facts list to exclude these going forward
			// Currently we ignore the country info for our pulling data from database as we check valid country below
			$events = $this->getEventsForDates($bdate + 150000,$ddate);
			if(count($events) > 0)
			{
				foreach($events as $eventKey=>$eventVal)
				{
					if(!isset($sourcesPrinted[$eventVal["id"]]))
					{
						if (($eventVal["gedcom_fact"] == "_MILI") &&
							($eventVal["startdate"] < $cmpDate) && ($cmpDate < $eventVal["enddate"]) &&
							(($eventVal["pl_lv1"] == NULL) || ($eventVal["pl_lv1"] == $tempCountry))) {
							$sourcesPrinted[$eventVal["id"]] = $eventVal;
						}
					}
				}
			}
		}

		// Next we are going to loop through all the PLAC facts found in the GEDCOM for this person (now stored in $factLookups)
		// For each one of these we generate an array for printing (to be stored in $sourcesPrinted)
		foreach($factLookups as $factLKey=>$factLValue)
		{
			$tempVal = trim($factLValue);

			// In the database schema there are events with start and end dates, so if the individual
			// was alive during these periods then there may be a missing fact
			// Note that military facts should assume that the individual was old enough to
			// participate, so in the case of events with gedcom_fact="_MILI" the birth date should be
			// minimum of real birth date + 15 years
			$events = $this->getEventsForDates($bdate,$ddate,"",$tempVal);
			if(count($events) > 0)
			{
				foreach($events as $eventKey=>$eventVal)
				{
					if(!isset($sourcesPrinted[$eventVal["id"]]))
					{
						$sourcesPrinted[$eventVal["id"]] = $eventVal;
						$tasktitle = htmlentities($eventVal["description"]);
						$taskid = $this->task_check($tasktitle, $person->getXref());

						if (!$taskid) // if the task_check passes, create a check box
						{
							$out .= '<tr ><td width="20" class="optionbox">';
							$out .= '<input type="checkbox" name="missingName[]" value="'.$tasktitle.'" />';
							$out .= '<td class="optionbox">'.$eventVal["description"];
							$out .= "</td></tr>";

						} else // if not allow user to view the already created task
							$out .= '<tr><td width="20" class="optionbox"><a href="module.php?mod=research_assistant'.
								'&amp;action=viewtask&amp;taskid='.$taskid.'">'.$pgv_lang["ra_view"].'</a></td><td class="optionbox">'.
								$tasktitle.'</td></tr>'."\n";
					}
				}
			}
		}

		// Next we are going to loop through all the events from between birth and death for this person (stored in $genericEvents)
		// For each one of these we generate an array for printing (to be stored in $sourcesPrinted)
		// Note there is a Special case if both birth and death dates are not specified then set $genericEvents = null array;
		// to ignore these dates otherwise we will throw a fatal exception
		if ((strlen($bdate) > 0) || (strlen($ddate) > 0)) {
			$genericEvents = $this->getEventsForDates($bdate,$ddate);

			// Strip out military events from returned data as we will do a special extract for them
			// as _MILI events are special cases so the minimum age should be 15 to return them
			$nonmiligenericEvents = array();
			foreach($genericEvents as $gKey=>$gVal)
			{
				if (!isset($sourcesPrinted[$gVal["id"]]) && ($gVal["gedcom_fact"]=="_MILI")) continue;
				else $nonmiligenericEvents = array_merge($nonmiligenericEvents,array($gKey=>$gVal));
			}
			$miligenericEvents = array();
			$genericEvents2 = $this->getEventsForDates($bdate + 150000,$ddate);

			foreach($genericEvents2 as $gKey=>$gVal)
			{
				if (!isset($sourcesPrinted[$gVal["id"]]) && ($gVal["gedcom_fact"]!="_MILI")) continue;
				else $miligenericEvents = array_merge($miligenericEvents,array($gKey=>$gVal));
			}
			if ((count($nonmiligenericEvents)==0) && (count($miligenericEvents)==0)) { $genericEvents = array(); } // do nothing 
			else if ((count($nonmiligenericEvents)>0) && (count($miligenericEvents)>0)) $genericEvents = array_merge($nonmiligenericEvents,$miligenericEvents);
			else if (count($nonmiligenericEvents)>0) $genericEvents = $nonmiligenericEvents;
			else $genericEvents = $miligenericEvents;
			
		}
		else 	$genericEvents = array();

		$lastPlace = null;
		foreach($genericEvents as $gKey=>$gVal)
		{
			if(!isset($sourcesPrinted[$gVal["id"]]))
			{
				$closest = null;
				$offset = null;
				$place = null;
				foreach($tempDates as $tKey=>$tVal)
				{
					$tempDate = $tVal->getDate();
					$tempPlace = $tVal->getPlace();
					$tempDate = $tempDate->date1;

					if (is_object($tempDate)) {
						$sortdate = $tempDate->y;
						$sortdate.=($tempDate->m) ? $tempDate->Format('m') : '00';
						$sortdate.=($tempDate->d) ? $tempDate->Format('d') : '00';
					} else $sortdate = '';

					$place = trim($tempPlace);
					if(empty($closest))
					{
						$closest = $sortdate;
						$place = $tempPlace;
						$lastPlace = $place;
					}
					else
					{
						$temp = $closest;
						$closest = $this->determineClosest($closest,$sortdate,$gVal["startdate"]);
						if($closest != $temp && !empty($tempPlace))
						{
							$place = $tempPlace;
							$lastPlace = $place;
						}
					}
				}
				$sourcesPrinted[$gVal["id"]] = $gVal;
				$tasktitle = htmlentities($gVal["description"]);
				$taskid = $this->task_check($tasktitle, $person->getXref());
				if (!$taskid) // if the task_check passes, create a check box
				{
					$out .= '<tr ><td width="20" class="optionbox">';
					$out .= '<input type="checkbox" name="missingName[]" value="'.$tasktitle.' '.$place.'" />';
					$out .= '<td class="optionbox">'.$gVal["description"];
					if(empty($place))
					{
						if(!empty($lastPlace))
						{
							$tempOut = $pgv_lang["ra_TheMostLikely"];
							$tempOut = str_replace("%INF_FACT%",$lastPlace,$tempOut);

							$out .= '<br/>'.$tempOut;
						}
					}
					else
					{
						$tempOut = $pgv_lang["ra_TheMostLikely"];
						$tempOut = str_replace("%INF_FACT%",$place,$tempOut);

						$out .= '<br/>'.$tempOut;
					}
					$out .= '</td></tr>';
				} else // if not allow user to view the already created task
					$out .= '<tr><td width="20" class="optionbox"><a href="module.php?mod=research_assistant'.
						'&amp;action=viewtask&amp;taskid='.$taskid.'">'.$pgv_lang["ra_view"].'</a></td><td class="optionbox">'.
						$tasktitle.'</td></tr>'."\n";
			}
		}

		// Create the selection box and add all the folder names and values
		$out .= '<tr><td class="optionbox" colspan="2" align="center"><h5>'.
			$pgv_lang["ra_Folder"].":&nbsp;&nbsp;".'</h><select name="folder">';
		$out .= $this->getFolders();  // used to be folder_search() but this restricted to Root level folders only
		$out .= '</select></td></tr>';
		$out .= '<tr><td colspan="2" class="topbottombar"><input type="submit" value="';
		$out .= $pgv_lang["ra_AddTask"].'" /></td></tr></table></form>'."\n";
		$out .= '</td><td width="5%"><br /></td>'."\n".'<td valign="top">';

		// Beginning of the auto search feature which gets dynamically populated with an
		// individuals information to be sent to an external site (like ancestry, familySearch, ...)
		$out .= "\n".'<table border="0"><tr>';
		$out .= '<td align="center" class="topbottombar" colspan="2" height="50%"><b>'.
			print_help_link("ra_auto_search_help", "qm", '', false, true).'<b>'.$pgv_lang['ra_auto_search_text'].'</b>';
		$out .= '</td></tr><tr>';
		$out .= '<td class="topbottombar">';
		$out .= '<form name="selector" action="" method="post" onsubmit="return false;">';
		$out .= '<select name="cbosite" onchange="search_selector('."'".$person->getXref()."'".');">';
		$out .= ' '.$this->autoSearchFormOptions().'</select></form>';
		$out .= '</td></tr><tr><td>'."\n";
		$out .= '<div id="searchdiv">';
		foreach ($this->sites as $file=>$value) break;

		require_once PGV_ROOT.'modules/research_assistant/search_plugin/'.$file;
		$autosearch=new AutoSearch();
		$out .=  $autosearch->options();
		$out .= '</div></td></tr></table></td></tr></table>';

		// Beginning of the comments feature processing here

		// If the last command was a delete_comment then remove that comment from the database
		// before proceeding with printing the comments table
		if (!empty($_REQUEST['action']) && $_REQUEST['action']=='delete_comment' && !empty($_REQUEST['uc_id'])) {
			PGV_DB::prepare("DELETE FROM {$TBLPREFIX}user_comments WHERE uc_id=?")
				->execute(array($_REQUEST['uc_id']));
		}
		$out .= '<br /><br />
			<table width="50%" align="center"><tr><td class="topbottombar">'.
				print_help_link("ra_comments_help", "qm", '', false, true).$pgv_lang["ra_comments"].'</td></tr>';
		$out .= '<tr><td class="optionbox">';

		// Display comments
		$rows=
			PGV_DB::prepare("SELECT uc_id, uc_username, uc_datetime, uc_comment FROM 
				{$TBLPREFIX}user_comments WHERE uc_f_id=? AND uc_p_id=? ORDER BY uc_datetime DESC")
			->execute(array(PGV_GED_ID, $person->getXref()))
			->fetchAll();
		// If no comments then print placeholder block header
		if (count($rows) == 0) {
			$out .= '<table class="blockheader" cellspacing="0" cellpadding="0">
				<tr>
					<td class="blockh1" >&nbsp;</td>
					<td class="blockh2" >
						<div class="blockhc"></div>
					</td>
					<td class="blockh3">&nbsp;</td>
				</tr>
				</table>';
		}
		foreach ($rows as $row) {
			$date=timestamp_to_gedcom_date($row->uc_datetime);
			$out .= '<div class="blockcontent"><div class="person_box" id="comment1"><span class="news_title">' .
				$row->uc_username.'' . // INSERT username
				'</span><br /><span class="news_date">' .
				$date->Display(false).' - '. date("g:i:s A",(int)$row->uc_datetime). // INSERT datetime
				'</span><br /><br />' .
				nl2br($row->uc_comment).
				'<hr size="1" />';

			if(PGV_USER_IS_ADMIN || PGV_USER_NAME==$row->uc_username) {
				// If permitted then display the edit and delete comment options
				$out .= '<a href="javascript:;" onclick="editcomment('.$row->uc_id.', \''.$person->getXref().'\'' . ')">'.$pgv_lang["ra_edit"].'</a>';
				$out .= ' | ';
				$out .= '<a href="" onclick="confirm_deletecomment(\''.$pgv_lang["ra_comment_delete_check"].'\', ' .$row->uc_id.', \''.$person->getXref().'\'); return false;">'.$pgv_lang["ra_delete"].'</a>';
			}
			$out .= '<br /></div></div><br />';
		}
		$out .= '</td></tr><tr><td class="topbottombar">';
		$out .= '<form action="" onsubmit="return false;">	<input type="button" value="'.
			$pgv_lang["ra_add_new_comment"].'" onclick="window.open(\'module.php?mod=research_assistant&action=editcomment&pid='.$person->getXref().'\', \'\',\'top=50,left=50,width=500,height=350,resizable=1,scrollbars=1\');"></form>';
		$out .= '</td></tr></table>';
		$out .= "\n\t".'<br /><br />';

		// Return the goods
		return $out;
	}

	/*
	 * This will return an array of comma delimited lists of all the PLAC facts for a person
	 *
	 * @person The object for the person you are looking for.
	 * @return An array containing comma delimited lists of all the PLAC facts found in the GEDCOM for this person.
	 */
	function getPlacesFromPerson($person){
		/*@var $person Person*/
		//Get the GEDCOM for the person
		$personGedcom = $person->getGedcomRecord();
		//Get all the Places for that person
		preg_match_all("/2 PLAC (.*)/",$personGedcom,$places,PREG_SET_ORDER);

		$returnPlaces = array();
		for($i = 0; $i < count($places);$i++)
		{
			if(!in_array($places[$i][1],$returnPlaces))
			{
				$returnPlaces[] = $places[$i][1];
			}
		}
		return $returnPlaces;
	}

	function decideInferSentence($percentage,$fact,$value)
	{
		global $pgv_lang;
		if (!isset($value) || empty($value)) $value = "";
		if($fact == "BIRT:PLAC")
		{
			if ((int)$percentage == 0) $percentage = "";
			else $percentage = sprintf("%.2f%%",$percentage * 100);

			$tempOut = $pgv_lang["ra_InferIndvBirthPlac"];
			$tempOut = str_replace("%PERCENT%",$percentage,$tempOut);
			$tempOut = str_replace("%INF_FACT%",$value,$tempOut);
			return $tempOut;
		}
		if($fact == "DEAT:PLAC")
		{
			if ((int)$percentage == 0) $percentage = "";
			else $percentage = sprintf("%.2f%%",(int) $percentage * 100);

			$tempOut = $pgv_lang["ra_InferIndvDeathPlac"];
			$tempOut = str_replace("%PERCENT%",$percentage,$tempOut);
			$tempOut = str_replace("%INF_FACT%",$value,$tempOut);
			return $tempOut;
		}
		if($fact == "SURN")
		{
			if ((int)$percentage == 0) $percentage = "";
			else $percentage = sprintf("%.2f%%",$percentage * 100);

			$tempOut = $pgv_lang["ra_InferIndvSurn"];
			$tempOut = str_replace("%PERCENT%",$percentage,$tempOut);
			$tempOut = str_replace("%INF_FACT%",$value,$tempOut);
			return $tempOut;
		}
		if($fact == "MARR:PLAC")
		{
			if ((int)$percentage == 0) $percentage = "";
			else $percentage = sprintf("%.2f%%",$percentage * 100);

			$tempOut = $pgv_lang["ra_InferIndvMarriagePlace"];
			$tempOut = str_replace("%PERCENT%",$percentage,$tempOut);
			$tempOut = str_replace("%INF_FACT%",$value,$tempOut);
			return $tempOut;
		}
		if($fact == "GIVN")
		{
			if ((int)$percentage == 0) $percentage = "";
			else $percentage = sprintf("%.2f%%",$percentage * 100);

			$tempOut = $pgv_lang["ra_InferIndvGivn"];
			$tempOut = str_replace("%PERCENT%",$percentage,$tempOut);
			$tempOut = str_replace("%INF_FACT%",$value,$tempOut);
			return $tempOut;
		}
	}


	/**
	 * function to handle fact edits/updates for task completion forms
	 * this function is hooked in from the edit_interface.php file.
	 * It requires at least that a ctype variable be put on the request
	 */
	function edit_fact() {
		global $factarray, $pgv_lang, $tag, $MULTI_MEDIA;

		if ($_REQUEST['ctype']=='add' && !empty($_REQUEST['fact'])) {
			$fact = $_REQUEST['fact'];
			if (!empty($_REQUEST['type'])) $type = $_REQUEST['type'];
			else $type='indi';

			init_calendar_popup();
			print "<form method=\"post\" action=\"edit_interface.php\" enctype=\"multipart/form-data\">\n";
			print "<input type=\"hidden\" name=\"action\" value=\"mod_edit_fact\" />\n";
			print "<input type=\"hidden\" name=\"mod\" value=\"research_assistant\" />\n";
			print "<input type=\"hidden\" name=\"ctype\" value=\"update\" />\n";
			print "<input type=\"hidden\" name=\"type\" value=\"$type\" />\n";
			print "<br /><input type=\"submit\" value=\"".$pgv_lang["ra_add"]."\" /><br />\n";
			print "<table class=\"facts_table\">";
			create_add_form($fact);
			print "</table>";

			if ($fact!="OBJE") {
				if ($fact!="NOTE") print_add_layer("NOTE");
				if ($fact!="REPO") print_add_layer("OBJE");
			}

			print "<br /><input type=\"submit\" value=\"".$pgv_lang["ra_add"]."\" /><br />\n";
			print "</form>\n";
		}
		if ($_REQUEST['ctype']=='edit' && !empty($_REQUEST['factrec'])) {
			$gedrec = $_REQUEST['factrec'];
			init_calendar_popup();
			print "<form method=\"post\" action=\"edit_interface.php\" enctype=\"multipart/form-data\">\n";
			print "<input type=\"hidden\" name=\"action\" value=\"mod_edit_fact\" />\n";
			print "<input type=\"hidden\" name=\"ctype\" value=\"update\" />\n";
			print "<input type=\"hidden\" name=\"mod\" value=\"research_assistant\" />\n";
			print "<br /><input type=\"submit\" value=\"".$pgv_lang["ra_save"]."\" /><br />\n";
			print "<table class=\"facts_table\">";
			$level0type = "INDI";
			create_edit_form($gedrec, 0, $level0type);
			print "</table>";
			print_add_layer("NOTE");
			print_add_layer("OBJE");

			print "<br /><input type=\"submit\" value=\"".$pgv_lang["ra_save"]."\" /><br />\n";
			print "</form>\n";
		}
		else if ($_REQUEST['ctype']=="update") {
			$factrec = handle_updates('');
			if (!empty($_REQUEST['type'])) $type = $_REQUEST['type'];
			else $type='indi';
			//print "<pre>$factrec</pre>";
			print '<script type="text/javascript">window.opener.paste_edit_data(\''.preg_replace("/\r?\n/", "\\r\\n", $factrec).'\', \''.$factarray[$tag[0]].'\', \''.$type.'\'); window.close();</script>';
		}
	}

	/**
	 * delete all facts associated with the given task id
	 * @param string $taskid the taskid to delete facts for
	 * @param GedcomRecord $record the record to look in
	 * @return string  the updated record without the associated facts
	 */
	function deleteRAFacts($taskid, &$record) {
		$indirec = $record->getGedcomRecord();
		if (preg_match("/\d _RATID ".$taskid."/", $indirec)>0) {
			$lines = preg_split("/[\r\n]/", $indirec);
			$newrec = $lines[0]."\r\n";
			$subrecs = get_all_subrecords($indirec, '', false, false, false);
			foreach($subrecs as $i=>$factrec) {
				if (preg_match("/\d _RATID ".$taskid."/", $factrec)==0) {
					$newrec .= trim($factrec)."\r\n";
				}
			}
			$indirec = $newrec;
		}
		return $indirec;
	}
}
?>
