<?php
/**
 * Common functions for the Search Plug-in of the Auto search Assistant
 * In general based on passed in array of fieldnames used in the form of the
 * URL passed.  The form may only need a few fields, or fields joined in
 * differnt ways, but Fieldnames available in the array generally include:
 * 	given names (variable: 'givenname')
 * 	surname (variable: 'surname')
 *	given names & surname (variable: 'fullname')
 * 	birth year (variable: 'byear')
 * 	birth location (variable: 'bloc')
 * 	death year (variable: 'dyear')
 * 	death location (variable: 'dloc')
 *	gender of given name (variable: 'gender')
 *	country of given name (variable: 'country')
 * 	father's full name (variable: 'ffullname')
 *	fathers first name (variable: 'fgivennames')
 *	fathers last name (variable: 'fsurname')
 * 	mother's name (variable: 'mfullname')
 *	mothers first name (variable: 'mgivennames')
 *	mothers last name (variable: 'msurname')
 * 	spouse's name (variable: 'sfullname')
 *	spouses first name (variable: 'sgivennames')
 *	spouses last name (variable: 'ssurname')
 *	marriage year (variable: 'myear')
 * 	keywords (variable: 'keywords')
 * 
 * Typical use of the fieldnames in the arrays are fieldname equals the value
 * 'function' followed by the variable.  First the fieldname is extracted
 * and then display as $pgv_lang["ra_autosearch_".fieldname] and if the next
 * in the same array is 'extra' it is used to automatically provide a checked
 * variable value.  Then the fieldname is used as a parameter
 * in the URL with value of the variable (i.e. url&fieldname=variablevalue)
 * where in this file there is a function with the variables name that knows
 * how to extract the value of the variable.  The resultant URL consists of 
 * the base URL plus numerous fieldname=value pairs separated by ampersands.
 *
 * If the first value of array is not 'function' but 'concatenate' then the
 * next two pairs in the array will displayed together (separated by a space)
 * but when the parameters are built it uses the concatenate fieldname equal
 * to the first variablevalue, the separator that follows the 'concatentate' in
 * the array (ie typically '_' as needed by the destination URL), then the
 * second variablevalue.
 *
 * phpGedView: Genealogy Viewer
 * Copyright (C) 2002 to 2024 PGV Development Team.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * @version $Id: base_autosearch.php 7377 2024-10-18 13:50:13Z canajun2eh $
 * @package PhpGedView
 * @author Greg Roach
 * @subpackage Research Assistant
 */

if (!defined('PGV_PHPGEDVIEW')) {
	header('HTTP/1.0 403 Forbidden');
	exit;
}

require_once PGV_ROOT.'includes/classes/class_person.php';

class Base_AutoSearch {
	private $title =null;
	private $url   =null;
	private $method=null;
	private $fields=null;

	// Constructor simply defines the fields to be used
	function __construct($title, $plugin, $url, $method, $fields) {
		$this->title =$title;
		$this->plugin =$plugin;
		$this->url   =$url;
		$this->method=$method;
		$this->fields=$fields;
	}

	// Generate a form to show the search options
	function options() {
		global $pgv_lang;

		$pid = safe_GET_xref('pid');
		if (empty($pid)) $pid = safe_POST_xref('pid');
		$person=Person::getInstance($pid);

		$autosearchfrmAction = "module.php?mod=research_assistant&amp;action=auto_search&amp;searchtype=".$this->title;
		$html ='<form name="autosearch" action="'.$autosearchfrmAction.'" target="_blank" method="post">';
		$html.='<input type="hidden" name="mod"        value="research_assistant" />';
		$html.='<input type="hidden" name="action"     value="auto_search" />';
		$html.='<input type="hidden" name="searchtype" value="'.$this->title.'" />';
		$html.='<table>';
		foreach ($this->fields as $field=>$settings) {
			if (!array_key_exists('extra', $settings)) {
				$settings['extra']='';
			}
			if (array_key_exists('concatenate', $settings)) continue;
			$html.='<tr><td class="descriptionbox wrap width33">'.$pgv_lang['ra_autosearch_'.$settings['function']].'</td>';
			$html.='<td class="optionbox wrap width66"><input type="checkbox" name="enable_'.$settings['function'].'" value="Y" ';
			$editFunction = $settings['function'];
			$edit = $this->$editFunction($person);
			if (strip_tags($edit)) {
				if ($settings['extra']) {
					$html.=$settings['extra'];
				}
			} else {
				$html.='disabled="true"';
			}
			$html.='/> '.$edit.'</td></tr>';
		}
		$html.='<tr><td class="optionbox" colspan="2" align="center">';
		$pgv_lang["ra_autosearch_plugin_name"] = $this->plugin;
		$html.=print_text($pgv_lang["ra_autosearch_plugin"],0,2).'</td></tr>';
		$html.='<tr><td class="topbottombar" colspan="2" align="center">';
		$html.='<input type="submit" value="'.$pgv_lang['ra_autosearch_search'].'" /></td></tr></table></form>';
		return $html;
	}

	// Execute the search by building the url with the list of enabled fields
	// If the field is concatenate and the next field is enabled, the next field
	// will be concatentated with the following field regardless of whether that
	// field is 
	function process() {
		$params=array();
		$concNextTwo = 0;
		$concSeparator = '';
		$concString = '';
		
		foreach ($this->fields as $field=>$settings) {
			if (array_key_exists('concatenate', $settings)) {
				$concNextTwo = 1;
				$concSeparator = $settings['concatenate'];
				continue;
			}
			else if ($concNextTwo == 1) {
				$concString = $field.'=';
				if (safe_POST_bool('enable_'.$settings['function'])) {
					$concString .= urlencode(safe_POST($settings['function']));
					$concNextTwo = 2;
					continue;
				}
				else {
					$concNextTwo = 0;
					$concString = '';
					continue;
				}
			}
			else if ($concNextTwo == 2) {
				$concString2 = urlencode(safe_POST($settings['function']));
				if (strlen($concString2) != 0) $concString = $concString.$concSeparator.$concString2;
				$params[] = $concString;
				$concNextTwo = 0;
				$concString = '';
				continue;
			}
			else if (safe_POST_bool('enable_'.$settings['function'])) {
				$params[] = $field.'='.urlencode(safe_POST($settings['function']));
			}
		}
		// TODO Handle forms that use POST as well as GET
		header('Location: '.$this->url.join('&', $params));
		exit;
	}

	// Functions to generate each search parameter
	function gender($person) {
		return '<input type="hidden" name="gender" value="'.$person->getSex().'">'.$person->getSex().$person->getSexImage();
	}

	function givenname($person, $inputname='givenname') {
		$all_givn=array();
		foreach ($person->getAllNames() as $name) {
			if ($name['givn']!='@P.N.') {
				$all_givn[]=htmlspecialchars(strip_tags($name['givn']));
			}
		}
		$all_givn=array_unique($all_givn);
		if (count($all_givn)==1) {
			return '<input type="hidden" name="'.$inputname.'" value="'.$all_givn[0].'">'.$all_givn[0];
		} else {
			$html='<select name="'.$inputname.'">';
			foreach ($all_givn as $givn) {
				$html.='<option value="'.$givn.'">'.$givn.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	// Pull the first name out of the list of given names and return that as the firstname
	function firstname($person, $inputname='firstname') {
		$all_givn=array();
		foreach ($person->getAllNames() as $name) {
			if ($name['givn']!='@P.N.') {
				if (count(explode(" ", htmlspecialchars(strip_tags($name['givn'])))) > 1)
					list($first_givn,$all_middlen) = explode(" ",htmlspecialchars(strip_tags($name['givn'])));
				else if (count(explode(" ", htmlspecialchars(strip_tags($name['givn'])))) == 1)
					$first_givn = htmlspecialchars(strip_tags($name['givn']));
				$all_givn[]=$first_givn;
			}
		}
		$all_givn=array_unique($all_givn);
		if (count($all_givn)==1) {
			return '<input type="hidden" name="'.$inputname.'" value="'.$all_givn[0].'">'.$all_givn[0];
		} else {
			$html='<select name="'.$inputname.'">';
			foreach ($all_givn as $givn) {
				$html.='<option value="'.$givn.'">'.$givn.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	// Throw out the first name out of the list of given names and return the remainder as the middlename
	function middlename($person, $inputname='middlename') {
		$all_middle=array();
		foreach ($person->getAllNames() as $name) {
			if ($name['givn']!='@P.N.') {
				if (count(explode(" ", htmlspecialchars(strip_tags($name['givn'])))) > 1)
					list($first_givn,$the_middle) = explode(" ",htmlspecialchars(strip_tags($name['givn'])),2);
				else 
					$the_middle = '';
				$all_middle[]=$the_middle;
			}
		}
		$all_middle=array_unique($all_middle);
		if (count($all_middle)==1) {
			return '<input type="hidden" name="'.$inputname.'" value="'.$all_middle[0].'">'.$all_middle[0];
		} else {
			$html='<select name="'.$inputname.'">';
			foreach ($all_middle as $mdle) {
				$html.='<option value="'.$mdle.'">'.$mdle.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	// Pull the surname field from a person and return that as their last name
	function surname(&$person, $inputname='surname') {
		$all_surn=array();
		foreach ($person->getAllNames() as $name) {
			if ($name['surname']!='@N.N.') {
				$all_surn[]=htmlspecialchars(strip_tags($name['surname']));
			}
		}
		$all_surn=array_unique($all_surn);
		if (count($all_surn)==1) {
			return '<input type="hidden" name="'.$inputname.'" value="'.$all_surn[0].'">'.$all_surn[0];
		} else {
			$html='<select name="'.$inputname.'">';
			foreach ($all_surn as $surn) {
				$html.='<option value="'.$surn.'">'.$surn.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	// Note that the fullname function separates first and multiple middle names each with plus signs 
	// and separates the given names from the surname by an underscore as these separators
	// are needed for numerous search websites to differentiate the different parts of names for searches

	function fullname(&$person, $inputname='fullname') {
		$all_full=array();
		foreach ($person->getAllNames() as $name) {
			if ($name['givn']!='@N.N.' && $name['surname']!='@N.N.') {
				$all_full[]=htmlspecialchars("{$name['givn']}")."_".htmlspecialchars("{$name['surname']}");
			}
		}
		$all_full=array_unique($all_full);
		if (count($all_full)==1) {
			return '<input type="hidden" name="'.$inputname.'" value="'.$all_full[0].'">'.htmlspecialchars("{$name['givn']} {$name['surname']}");
		} else {
			$html='<select name="'.$inputname.'">';
			foreach ($all_full as $full) {
				$html.='<option value="'.$full.'">'.str_replace("_"," ",$full).'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	function fgivennames($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getHusband()) {
			return $this->givenname($parents->getHusband(), 'fgivennames');
		} else {
			return '<input type="hidden" name="fgivennames" value="">';
		}
	}

	function fsurname($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getHusband()) {
			return $this->surname($parents->getHusband(), 'fsurname');
		} else {
			return '<input type="hidden" name="fsurname" value="">';
		}
	}

	function ffullname($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getHusband()) {
			return $this->fullname($parents->getHusband(), 'ffullname');
		} else {
			return '<input type="hidden" name="ffullname" value="">';
		}
	}

	function mgivennames($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getWife()) {
			return $this->givenname($parents->getWife(), 'mgivennames');
		} else {
			return '<input type="hidden" name="mgivennames" value="">';
		}
	}

	function msurname($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getWife()) {
			return $this->surname($parents->getWife(), 'msurname');
		} else {
			return '<input type="hidden" name="msurname" value="">';
		}
	}

	function mfullname($person) {
		$parents=$person->getPrimaryChildFamily();
		if ($parents && $parents->getWife()) {
			return $this->fullname($parents->getWife(), 'mfullname');
		} else {
			return '<input type="hidden" name="mfullname" value="">';
		}
	}

	function sgivennames($person) {
		$spouse=$person->getCurrentSpouse();
		if ($spouse) {
			return $this->givenname($spouse, 'sgivennames');
		} else {
			return '<input type="hidden" name="sgivennames" value="">';
		}
	}

	function ssurname($person) {
		$spouse=$person->getCurrentSpouse();
		if ($spouse) {
			return $this->surname($spouse, 'ssurname');
		} else {
			return '<input type="hidden" name="ssurname" value="">';
		}
	}

	function sfullname($person) {
		$spouse=$person->getCurrentSpouse();
		if ($spouse) {
			return $this->fullname($spouse, 'sfullname');
		} else {
			return '<input type="hidden" name="sfullname" value="">';
		}
	}

	function byear($person) {
		$birth=$person->getEstimatedBirthDate();
		$byear=$birth->gregorianYear();

		// Note that if the birth year is empty then getEstimatedBirthDate could return a bogus
		// date based on the death date, so we need to test for this special case
		if ((($person->getBirthDate()->gregorianYear()) == 0) || $byear==0) {
			$byear='';
		}
		return '<input type="hidden" name="byear" value="'.$byear.'">'.$byear;
	}

	function dyear($person) {
		$death=$person->getEstimatedDeathDate();
		$dyear=$death->gregorianYear();

		// Note that if the death year is empty then getEstimatedDeathDate could return a bogus
		// date based on the birth date, so we need to test for this special case
		if ((($person->getDeathDate()->gregorianYear()) == 0) || $dyear==0) {
			$dyear='';
		}
		return '<input type="hidden" name="dyear" value="'.$dyear.'">'.$dyear;
	}

	function myear($person) {
		$myears=array();
		foreach ($person->getSpouseFamilies() as $family) {
			$mdate=$family->getMarriageDate();
			if ($mdate->isOK()) {
				$myears[]=$mdate->gregorianYear();
			} else {
				$myears[]='';
			}
		}
		$myears=array_unique($myears);
		if (count($myears)==1) {
			return '<input type="hidden" name="myear" value="'.$myears[0].'">'.$myears[0];
		} else {
			$html='<select name="myear">';
			foreach ($myears as $myear) {
				$html.='<option value="'.$myear.'">'.$myear.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	function bloc($person) {
		$place=htmlspecialchars($person->getBirthPlace());
		return '<input type="hidden" name="bloc" value="'.$place.'">'.$place;
	}

	function dloc($person) {
		$place=htmlspecialchars($person->getDeathPlace());
		return '<input type="hidden" name="dloc" value="'.$place.'">'.$place;
	}

	function mloc($person) {
		$mplaces=array();
		foreach ($person->getSpouseFamilies() as $family) {
			$mplaces[]=htmlspecialchars($family->getMarriagePlace());
		}
		$mplaces=array_unique($mplaces);
		if (count($mplaces)==1) {
			return '<input type="hidden" name="mloc" value="'.$mplaces[0].'">'.$mplaces[0];
		} else {
			$html='<select name="mloc">';
			foreach ($mplaces as $mplace) {
				$html.='<option value="'.$mplace.'">'.$mplace.'</option>';
			}
			$html.='</select>';
			return $html;
		}
	}

	function keywords($person) {
		return '<input name="keywords">&zwnj;'; // zwnj prevents empty value being disabled
	}

	/*
	 * This will return an array of comma delimited lists of all the CEME facts for a person
	 *
	 * @person The object for the person you are looking for.
	 * @return An string containing of the BURI:PLAC fact found in the GEDCOM for this person.
	 */
	function cemetery($person) {
		//Get the GEDCOM for the person
		$personGedcom = $person->getGedcomRecord();
		//Get all the Places for that person
		preg_match_all("/1 BURI((.|\n)*)2 PLAC (.*)/",$personGedcom,$burialplac,PREG_SET_ORDER);
		if (isset($burialplac[0][3])) return $burialplac[0][3];
		else return "";
// Return Ceme entry - above returns burial location which is what findagrave wants
//		preg_match_all("/2 CEME (.*)/",$personGedcom,$cemeteries,PREG_SET_ORDER);
//		$returnCemes = array();
//		for($i = 0; $i < count($cemeteries);$i++)
//		{
//			if(!in_array($cemeteries[$i][1],$returnCemes))
//			{
//				return '<input type="hidden" name="cemetery" value="'.$cemeteries[$i][1].'">'.$cemeteries[$i][1];
//				$returnCemes[] = $cemeteries[$i][1];
//			}
//		}
//		return "";
	}

	function country($person) {
		global $countries;

		// Fetch a list of countries
		loadLangFile('pgv_country');
		asort($countries);

		// Move birth/death countries to top of list
		$bcountry=array_pop(preg_split('/ *, */', $person->getBirthPlace()));
		$dcountry=array_pop(preg_split('/ *, */', $person->getDeathPlace()));

		$html='<select name="country">';
		if ($bcountry && array_search($bcountry, $countries)) {
			$html.='<option value="'.htmlspecialchars(array_search($bcountry, $countries)).'">'.htmlspecialchars($bcountry).'</option>';
		}
		if ($dcountry && array_search($dcountry, $countries) && $bcountry!=$dcountry) {
			$html.='<option value="'.htmlspecialchars(array_search($dcountry, $countries)).'">'.htmlspecialchars($dcountry).'</option>';
		}
		foreach ($countries as $code=>$country) {
			if ($country!=$bcountry && $country!=$dcountry) {
				$html.='<option value="'.htmlspecialchars($code).'">'.htmlspecialchars($country).'</option>';
			}
		}
		$html.='</select>';
		return $html;
	}

}

?>
